<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class EstudiantesController extends Controller
{
    public function index(Request $request)
    {
        $annio = (int) $request->query('annio', date('Y'));
        $curso = trim((string) $request->query('curso', ''));
        $nombres = trim((string) $request->query('nombres', ''));
        $codigo = trim((string) $request->query('codigo', ''));
        $documento = trim((string) $request->query('documento', ''));

        [$baseTable, $baseDatosTable, $hasYearTable] = $this->resolveTablesByYear($annio);

        $rows = DB::table("{$baseTable} as e")
            ->leftJoin('sweb_salones as s', 's.id', '=', 'e.salon_id')
            ->leftJoin("{$baseDatosTable} as d", 'd.estudiante_id', '=', 'e.id')
            ->select(
                'e.id',
                'e.nombres',
                'e.apellido1',
                'e.apellido2',
                'e.documento',
                'e.numero_mat',
                'e.fecha_nac',
                'e.photo',
                'e.is_deudor',
                'e.contabilidad_id',
                'e.annio_pagado',
                's.nombre as curso_nombre',
                'd.madre',
                'd.madre_tel_1',
                'd.padre',
                'd.padre_tel_1'
            )
            ->when(! $hasYearTable, function ($query) use ($annio) {
                $query->whereIn('e.id', function ($sub) use ($annio) {
                    $sub->from('sweb_notas_v2')
                        ->select('estudiante_id')
                        ->where('annio', $annio)
                        ->distinct();
                });
            })
            ->when($curso !== '', function ($query) use ($curso) {
                $query->where('e.salon_id', (int) $curso);
            })
            ->when($nombres !== '', function ($query) use ($nombres) {
                $query->where(function ($sub) use ($nombres) {
                    $sub->where('e.nombres', 'like', "%{$nombres}%")
                        ->orWhere('e.apellido1', 'like', "%{$nombres}%")
                        ->orWhere('e.apellido2', 'like', "%{$nombres}%");
                });
            })
            ->when($codigo !== '', function ($query) use ($codigo) {
                $query->whereRaw("LPAD(COALESCE(NULLIF(e.contabilidad_id,0), e.id), 4, '0') LIKE ?", ["{$codigo}%"]);
            })
            ->when($documento !== '', function ($query) use ($documento) {
                $query->where('e.documento', 'like', "%{$documento}%");
            })
            ->orderBy('e.nombres')
            ->paginate(20)
            ->withQueryString();

        $cursos = DB::table('sweb_salones')
            ->select('id', 'nombre')
            ->orderBy('nombre')
            ->get();

        return view('admin.estudiantes.index', compact(
            'rows',
            'annio',
            'curso',
            'nombres',
            'codigo',
            'documento',
            'cursos'
        ));
    }

    public function show(Request $request, int $id)
    {
        $annio = (int) $request->query('annio', date('Y'));
        [$baseTable, $baseDatosTable] = $this->resolveTablesByYear($annio);

        $student = DB::table("{$baseTable} as e")
            ->leftJoin("{$baseDatosTable} as d", 'd.estudiante_id', '=', 'e.id')
            ->leftJoin('sweb_salones as s', 's.id', '=', 'e.salon_id')
            ->select(
                'e.*',
                'd.id as datos_id',
                'd.ciudad_nac',
                'd.grupo_sang',
                'd.salud_eps',
                'd.religion',
                'd.madre',
                'd.madre_id',
                'd.madre_tel_1',
                'd.madre_email',
                'd.padre',
                'd.padre_id',
                'd.padre_tel_1',
                'd.padre_email',
                'd.acudiente',
                'd.parentesco',
                'd.acudi_id',
                'd.acudi_tel_1',
                'd.acudi_email',
                's.nombre as curso_nombre'
            )
            ->where('e.id', $id)
            ->first();

        abort_unless($student, 404);

        $allSalones = DB::table('sweb_salones')
            ->select('id', 'nombre')
            ->orderBy('nombre')
            ->get();

        $firstName = trim((string) $student->nombres);
        $parts = preg_split('/\s+/', $firstName);
        $primerNombre = $parts[0] ?? '';
        $segundoNombre = implode(' ', array_slice($parts, 1));

        $familyMembers = collect([
            [
                'nombre' => $student->madre,
                'parentesco' => 'Madre',
                'documento' => $student->madre_id,
                'telefono' => $student->madre_tel_1,
                'email' => $student->madre_email,
                'codigo' => null,
            ],
            [
                'nombre' => $student->padre,
                'parentesco' => 'Padre',
                'documento' => $student->padre_id,
                'telefono' => $student->padre_tel_1,
                'email' => $student->padre_email,
                'codigo' => null,
            ],
            [
                'nombre' => trim(($student->nombres ?? '') . ' ' . ($student->apellido1 ?? '') . ' ' . ($student->apellido2 ?? '')),
                'parentesco' => 'Estudiante',
                'documento' => $student->documento,
                'telefono' => $student->telefono1,
                'email' => $student->email,
                'codigo' => 'WS-' . str_pad((string) ((int) ($student->contabilidad_id ?: $student->id)), 4, '0', STR_PAD_LEFT),
            ],
        ])->filter(fn ($m) => ! empty(trim((string) ($m['nombre'] ?? ''))))->values();

        return view('admin.estudiantes.show', compact(
            'annio',
            'student',
            'allSalones',
            'primerNombre',
            'segundoNombre',
            'familyMembers'
        ));
    }

    public function update(Request $request, int $id)
    {
        $annio = (int) $request->query('annio', date('Y'));
        [$baseTable, $baseDatosTable] = $this->resolveTablesByYear($annio);

        $data = $request->validate([
            'primer_nombre' => ['nullable', 'string', 'max:50'],
            'segundo_nombre' => ['nullable', 'string', 'max:100'],
            'primer_apellido' => ['nullable', 'string', 'max:30'],
            'segundo_apellido' => ['nullable', 'string', 'max:30'],
            'fecha_nac' => ['nullable', 'date'],
            'ciudad_nac' => ['nullable', 'string', 'max:50'],
            'tipo_dcto' => ['nullable', 'string', 'max:255'],
            'documento' => [
                'nullable',
                'string',
                'max:20',
                Rule::unique($baseTable, 'documento')->ignore($id),
            ],
            'eps' => ['nullable', 'string', 'max:50'],
            'tipo_sangre' => ['nullable', 'string', 'max:255'],
            'religion' => ['nullable', 'string', 'max:50'],
            'email' => ['nullable', 'email', 'max:100'],
            'sexo' => ['nullable', 'string', 'max:255'],
            'salon_id' => ['nullable', 'integer'],
            'is_habilitar_mat' => ['nullable', 'boolean'],
            'photo' => ['nullable', 'image', 'max:2048'],
        ]);

        $fullName = trim(($data['primer_nombre'] ?? '') . ' ' . ($data['segundo_nombre'] ?? ''));
        if ($fullName === '') {
            $fullName = trim((string) DB::table($baseTable)->where('id', $id)->value('nombres'));
        }

        $studentUpdate = [
            'nombres' => $fullName,
            'apellido1' => $data['primer_apellido'] ?? null,
            'apellido2' => $data['segundo_apellido'] ?? null,
            'fecha_nac' => $data['fecha_nac'] ?? null,
            'tipo_dcto' => $data['tipo_dcto'] ?? null,
            'documento' => $data['documento'] ?? null,
            'email' => $data['email'] ?? null,
            'sexo' => $data['sexo'] ?? null,
            'salon_id' => $data['salon_id'] ?? null,
            'is_habilitar_mat' => (int) ($data['is_habilitar_mat'] ?? 0),
            'updated_at' => Carbon::now(),
            'updated_by' => 1,
        ];

        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('students', 'public');
            $studentUpdate['photo'] = $path;
        }

        DB::table($baseTable)->where('id', $id)->update($studentUpdate);

        $datosUpdate = [
            'ciudad_nac' => $data['ciudad_nac'] ?? null,
            'grupo_sang' => $data['tipo_sangre'] ?? null,
            'salud_eps' => $data['eps'] ?? null,
            'religion' => $data['religion'] ?? null,
            'updated_at' => Carbon::now(),
            'updated_by' => 1,
        ];

        $existsDatos = DB::table($baseDatosTable)->where('estudiante_id', $id)->exists();
        if ($existsDatos) {
            DB::table($baseDatosTable)->where('estudiante_id', $id)->update($datosUpdate);
        } else {
            $datosUpdate['estudiante_id'] = $id;
            $datosUpdate['created_at'] = Carbon::now();
            $datosUpdate['created_by'] = 1;
            DB::table($baseDatosTable)->insert($datosUpdate);
        }

        Cache::forget('ui_academic_years');

        return redirect()
            ->route('admin.estudiantes.show', ['id' => $id, 'annio' => $annio])
            ->with('status', 'Estudiante actualizado correctamente.');
    }

    private function resolveTablesByYear(int $annio): array
    {
        $yearTable = 'sweb_estudiantes_' . $annio;
        $hasYearTable = Cache::remember("has_table_{$yearTable}", 3600, function () use ($yearTable) {
            return DB::table('information_schema.tables')
                ->whereRaw('table_schema = DATABASE()')
                ->where('table_name', $yearTable)
                ->exists();
        });

        $yearDatosTable = 'sweb_datosestud_' . $annio;
        $hasYearDatosTable = Cache::remember("has_table_{$yearDatosTable}", 3600, function () use ($yearDatosTable) {
            return DB::table('information_schema.tables')
                ->whereRaw('table_schema = DATABASE()')
                ->where('table_name', $yearDatosTable)
                ->exists();
        });

        $baseTable = $hasYearTable ? $yearTable : 'sweb_estudiantes';
        $baseDatosTable = $hasYearDatosTable ? $yearDatosTable : 'sweb_datosestud';

        return [$baseTable, $baseDatosTable, $hasYearTable];
    }
}
