<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Http\UploadedFile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class SectionController extends Controller
{
    public function show(string $section)
    {
        $allowed = [
            'general',
            'matriculas',
            'mensajeria',
            'academia',
            'calificaciones',
            'observador',
            'contabilidad',
        ];

        abort_unless(in_array($section, $allowed, true), 404);


        $titles = [
            'general' => 'General',
            'matriculas' => 'Matriculas',
            'mensajeria' => 'Mensajeria',
            'academia' => 'Academia',
            'calificaciones' => 'Calificaciones',
            'observador' => 'Observador',
            'contabilidad' => 'Contabilidad',
        ];

        return view('admin.section', [
            'title' => $titles[$section] ?? Str::title($section),
            'section' => $section,
        ]);
    }

    private function renderListasDeCurso()
    {
        $annio = (int) request('annio', date('Y'));
        $profesor = (string) request('profesor', '');
        $asignatura = (string) request('asignatura', '');
        $periodo = (string) request('periodo', '');
        $papel = (string) request('papel', 'Carta');

        $profesores = DB::table('dm_user')
            ->select('id', 'nombres', 'apellido1', 'apellido2')
            ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->where('is_active', 1)
            ->orderBy('nombres')
            ->orderBy('apellido1')
            ->get();

        $asignaturas = DB::table('sweb_asignaturas')
            ->select('id', 'nombre')
            ->where('is_active', 1)
            ->orderBy('nombre')
            ->get();

        $periodos = DB::table('sweb_periodos')
            ->select('id', 'periodo')
            ->orderBy('id')
            ->get()
            ->map(function ($p) {
                $numero = null;
                if (preg_match('/(\d+)/', (string) $p->periodo, $m)) {
                    $numero = (int) $m[1];
                }
                return (object) [
                    'id' => $p->id,
                    'numero' => $numero,
                    'label' => $numero ? ($numero . 'Â°') : (string) $p->periodo,
                ];
            });

        $assignTable = $this->resolveSalonAsignatProfesorTable($annio);

        $rows = DB::table('sweb_salones as s')
            ->leftJoin('sweb_grados as g', 'g.id', '=', 's.grado_id')
            ->leftJoin('sweb_secciones as sec', 'sec.id', '=', 'g.seccion_id')
            ->leftJoin('dm_user as d', 'd.id', '=', 's.director_id')
            ->where('s.is_active', 1)
            ->when($profesor !== '' || $asignatura !== '', function ($q) use ($assignTable, $profesor, $asignatura) {
                $q->whereExists(function ($sub) use ($assignTable, $profesor, $asignatura) {
                    $sub->from("{$assignTable} as sap")
                        ->selectRaw('1')
                        ->whereColumn('sap.salon_id', 's.id');

                    if ($profesor !== '') {
                        $sub->where('sap.user_id', (int) $profesor);
                    }
                    if ($asignatura !== '') {
                        $sub->where('sap.asignatura_id', (int) $asignatura);
                    }
                });
            })
            ->select(
                's.id',
                's.nombre',
                's.tot_estudiantes',
                DB::raw("COALESCE(sec.nombre, '') as nivel"),
                DB::raw("TRIM(CONCAT(COALESCE(d.nombres,''),' ',COALESCE(d.apellido1,''),' ',COALESCE(d.apellido2,''))) as director")
            )
            ->orderBy('s.position')
            ->orderBy('s.nombre')
            ->get();

        return view('admin.general.cursos.listas_curso', [
            'title' => 'Listas de Curso',
            'annio' => $annio,
            'profesor' => $profesor,
            'asignatura' => $asignatura,
            'periodo' => $periodo,
            'papel' => $papel,
            'profesores' => $profesores,
            'asignaturas' => $asignaturas,
            'periodos' => $periodos,
            'rows' => $rows,
        ]);
    }

    private function renderAcudientes()
    {
        $annio = (int) request('annio', date('Y'));
        $nombres = trim((string) request('nombres', ''));
        $documento = trim((string) request('documento', ''));
        $codigoEstudiante = trim((string) request('codigo_estudiante', ''));

        [$estudiantesTable, $datosTable] = $this->resolveStudentTablesByYear($annio);

        $rows = DB::table("{$datosTable} as d")
            ->join("{$estudiantesTable} as e", 'e.id', '=', 'd.estudiante_id')
            ->select(
                'd.id',
                DB::raw("TRIM(COALESCE(d.acudiente,'')) as nombre"),
                DB::raw("COALESCE(d.acudi_id, '') as documento"),
                DB::raw("COALESCE(NULLIF(d.acudi_tel_1,''), d.acudi_tel_2, '') as telefono"),
                DB::raw("COALESCE(d.acudi_email, '') as email"),
                'e.is_deudor',
                'e.id as estudiante_id',
                'e.contabilidad_id'
            )
            ->where('e.is_active', 1)
            ->whereRaw("TRIM(COALESCE(d.acudiente,'')) <> ''")
            ->when($nombres !== '', function ($q) use ($nombres) {
                $q->where('d.acudiente', 'like', "%{$nombres}%");
            })
            ->when($documento !== '', function ($q) use ($documento) {
                $q->where('d.acudi_id', 'like', "%{$documento}%");
            })
            ->when($codigoEstudiante !== '', function ($q) use ($codigoEstudiante) {
                $q->whereRaw("LPAD(COALESCE(NULLIF(e.contabilidad_id,0), e.id), 4, '0') LIKE ?", ["{$codigoEstudiante}%"]);
            })
            ->orderBy('d.acudiente')
            ->paginate(20)
            ->withQueryString();

        return view('admin.general.grupos.acudientes', [
            'annio' => $annio,
            'nombres' => $nombres,
            'documento' => $documento,
            'codigoEstudiante' => $codigoEstudiante,
            'rows' => $rows,
        ]);
    }

    private function renderTodosCursos()
    {
        $rows = DB::table('sweb_salones as s')
            ->leftJoin('sweb_grados as g', 'g.id', '=', 's.grado_id')
            ->leftJoin('sweb_secciones as sec', 'sec.id', '=', 'g.seccion_id')
            ->leftJoin('dm_user as d', 'd.id', '=', 's.director_id')
            ->where('s.is_active', 1)
            ->select(
                's.id',
                's.nombre',
                DB::raw("COALESCE(sec.nombre, '') as nivel"),
                DB::raw("COALESCE(s.tot_estudiantes, 0) as tot_estudiantes"),
                DB::raw("TRIM(CONCAT(COALESCE(d.nombres,''),' ',COALESCE(d.apellido1,''),' ',COALESCE(d.apellido2,''))) as director")
            )
            ->orderBy('s.position')
            ->orderBy('s.nombre')
            ->get();

        return view('admin.general.cursos.todos', [
            'title' => 'Cursos',
            'rows' => $rows,
        ]);
    }

    private function renderTodosGrados()
    {
        $q = trim((string) request('q', ''));
        $nivel = (string) request('nivel', '');
        $estado = (string) request('estado', 'activos');

        $rows = DB::table('sweb_grados as g')
            ->leftJoin('sweb_secciones as s', 's.id', '=', 'g.seccion_id')
            ->leftJoin('sweb_salones as sl', function ($join) {
                $join->on('sl.grado_id', '=', 'g.id')
                    ->where('sl.is_active', 1);
            })
            ->when($q !== '', function ($query) use ($q) {
                $query->where(function ($sub) use ($q) {
                    $sub->where('g.nombre', 'like', "%{$q}%")
                        ->orWhere('g.abrev', 'like', "%{$q}%");
                });
            })
            ->when($nivel !== '', function ($query) use ($nivel) {
                $query->where('g.seccion_id', (int) $nivel);
            })
            ->when($estado === 'activos', fn ($query) => $query->where('g.is_active', 1))
            ->when($estado === 'inactivos', fn ($query) => $query->where('g.is_active', 0))
            ->select(
                'g.id',
                'g.nombre',
                'g.abrev',
                'g.is_active',
                DB::raw("COALESCE(s.nombre, '') as nivel"),
                DB::raw("COALESCE(g.orden, 0) as orden"),
                DB::raw('COUNT(sl.id) as cursos_count')
            )
            ->groupBy('g.id', 'g.nombre', 'g.abrev', 'g.is_active', 's.nombre', 'g.orden')
            ->orderByRaw('CASE WHEN g.orden IS NULL THEN 9999 ELSE g.orden END ASC')
            ->orderBy('g.nombre')
            ->get();

        $niveles = DB::table('sweb_secciones')
            ->select('id', 'nombre')
            ->orderBy('nombre')
            ->get();

        return view('admin.general.grados.todos', [
            'title' => 'Grados',
            'rows' => $rows,
            'niveles' => $niveles,
            'q' => $q,
            'nivel' => $nivel,
            'estado' => $estado,
        ]);
    }

    public function showGeneralItem(string $item)
    {
        $allowed = [
            'estudiantes',
            'profesores',
            'grupos-familiares',
            'cambio-clave',
        ];

        abort_unless(in_array($item, $allowed, true), 404);

        $titles = [
            'estudiantes' => 'General - Estudiantes',
            'profesores' => 'General - Profesores',
            'grupos-familiares' => 'General - Grupos Familiares',
            'cambio-clave' => 'General - Cambio de Clave',
        ];

        return view('admin.section', [
            'title' => $titles[$item] ?? Str::title(str_replace('-', ' ', $item)),
            'section' => 'general.' . $item,
        ]);
    }

    public function showGeneralGruposItem(string $item)
    {
        $allowed = [
            'acudientes',
            'grupos-familiares',
        ];

        abort_unless(in_array($item, $allowed, true), 404);

        $titles = [
            'acudientes' => 'General - Grupos Familiares - Acudientes',
            'grupos-familiares' => 'General - Grupos Familiares',
        ];

        if ($item === 'acudientes') {
            return $this->renderAcudientes();
        }

        return view('admin.section', [
            'title' => $titles[$item] ?? Str::title(str_replace('-', ' ', $item)),
            'section' => 'general.grupos-familiares.' . $item,
        ]);
    }

    public function showGeneralCursosItem(string $item)
    {
        $allowed = [
            'todos',
            'listas-curso',
        ];

        abort_unless(in_array($item, $allowed, true), 404);

        if ($item === 'listas-curso') {
            return $this->renderListasDeCurso();
        }
        if ($item === 'todos') {
            return $this->renderTodosCursos();
        }

        return view('admin.section', [
            'title' => 'General - Cursos - Todos',
            'section' => 'general.cursos.todos',
        ]);
    }

    public function showGeneralGradosItem(string $item)
    {
        $allowed = [
            'todos',
        ];

        abort_unless(in_array($item, $allowed, true), 404);

        if ($item === 'todos') {
            return $this->renderTodosGrados();
        }

        return view('admin.section', [
            'title' => 'General - Grados - Todos',
            'section' => 'general.grados.todos',
        ]);
    }

    public function createGeneralGrado()
    {
        $secciones = DB::table('sweb_secciones')
            ->select('id', 'nombre')
            ->orderBy('nombre')
            ->get();

        return view('admin.general.grados.show', [
            'isCreate' => true,
            'grado' => (object) [
                'id' => null,
                'nombre' => '',
                'abrev' => '',
                'seccion_id' => null,
                'orden' => 0,
                'is_active' => 1,
            ],
            'secciones' => $secciones,
        ]);
    }

    public function storeGeneralGrado(Request $request)
    {
        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:50', Rule::unique('sweb_grados', 'nombre')],
            'abrev' => ['required', 'string', 'max:10', Rule::unique('sweb_grados', 'abrev')],
            'seccion_id' => ['nullable', 'integer', Rule::exists('sweb_secciones', 'id')],
            'orden' => ['nullable', 'integer', 'min:0', 'max:9999'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        DB::table('sweb_grados')->insert([
            'uuid' => (string) Str::uuid(),
            'nombre' => trim((string) $data['nombre']),
            'abrev' => trim((string) $data['abrev']),
            'seccion_id' => $data['seccion_id'] !== null ? (int) $data['seccion_id'] : null,
            'orden' => (int) ($data['orden'] ?? 0),
            'is_active' => (int) ($data['is_active'] ?? 0),
            'salon_default' => 0,
            'proximo_salon' => 0,
            'proximo_grado' => 0,
            'created_by' => 1,
            'updated_by' => 1,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        $id = (int) DB::getPdo()->lastInsertId();

        return redirect()
            ->route('admin.general.grados.show', ['id' => $id])
            ->with('status', 'Grado creado correctamente.');
    }

    public function showGeneralGradoDetalle(int $id)
    {
        $grado = DB::table('sweb_grados')
            ->select('id', 'nombre', 'abrev', 'seccion_id', 'orden', 'is_active')
            ->where('id', $id)
            ->first();

        abort_unless($grado, 404);

        $secciones = DB::table('sweb_secciones')
            ->select('id', 'nombre')
            ->orderBy('nombre')
            ->get();

        return view('admin.general.grados.show', [
            'isCreate' => false,
            'grado' => $grado,
            'secciones' => $secciones,
        ]);
    }

    public function updateGeneralGradoDetalle(Request $request, int $id)
    {
        $grado = DB::table('sweb_grados')->where('id', $id)->first();
        if (! $grado) {
            return redirect()
                ->route('admin.general.grados.item', ['item' => 'todos'])
                ->with('status', 'No se encontrÃ³ el grado.');
        }

        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:50', Rule::unique('sweb_grados', 'nombre')->ignore($id)],
            'abrev' => ['required', 'string', 'max:10', Rule::unique('sweb_grados', 'abrev')->ignore($id)],
            'seccion_id' => ['nullable', 'integer', Rule::exists('sweb_secciones', 'id')],
            'orden' => ['nullable', 'integer', 'min:0', 'max:9999'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        DB::table('sweb_grados')
            ->where('id', $id)
            ->update([
                'nombre' => trim((string) $data['nombre']),
                'abrev' => trim((string) $data['abrev']),
                'seccion_id' => $data['seccion_id'] !== null ? (int) $data['seccion_id'] : null,
                'orden' => (int) ($data['orden'] ?? 0),
                'is_active' => (int) ($data['is_active'] ?? 0),
                'updated_by' => 1,
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.general.grados.show', ['id' => $id])
            ->with('status', 'Grado actualizado correctamente.');
    }
    public function deleteGeneralGrado(int $id)
    {
        $grado = DB::table('sweb_grados')->where('id', $id)->first();
        if (! $grado) {
            return redirect()
                ->route('admin.general.grados.item', ['item' => 'todos'])
                ->with('status', 'No se encontrÃ³ el grado para eliminar.');
        }

        $inUseSalones = DB::table('sweb_salones')
            ->where('grado_id', $id)
            ->where('is_active', 1)
            ->exists();
        $inUseAsignaturas = DB::table('sweb_grado_asignat')
            ->where('grado_id', $id)
            ->exists();

        if ($inUseSalones || $inUseAsignaturas) {
            return redirect()
                ->route('admin.general.grados.item', ['item' => 'todos'])
                ->with('status', 'No se puede eliminar: el grado tiene cursos/asignaturas asociadas.');
        }

        DB::table('sweb_grados')->where('id', $id)->delete();

        return redirect()
            ->route('admin.general.grados.item', ['item' => 'todos'])
            ->with('status', 'Grado eliminado correctamente.');
    }

    public function showGeneralCursoDetalle(Request $request, int $id)
    {
        $annio = (int) $request->query('annio', date('Y'));

        $course = DB::table('sweb_salones')->where('id', $id)->first();
        abort_unless($course, 404);

        $docentes = DB::table('dm_user')
            ->select('id', 'nombres', 'apellido1', 'apellido2')
            ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->where('is_active', 1)
            ->orderBy('nombres')
            ->orderBy('apellido1')
            ->get();

        $students = DB::table('sweb_estudiantes as e')
            ->select('e.id', 'e.nombres', 'e.apellido1', 'e.apellido2', 'e.photo')
            ->where('e.salon_id', $id)
            ->where('e.is_active', 1)
            ->orderBy('e.apellido1')
            ->orderBy('e.apellido2')
            ->orderBy('e.nombres')
            ->get();

        return view('admin.general.cursos.show', [
            'annio' => $annio,
            'course' => $course,
            'docentes' => $docentes,
            'students' => $students,
        ]);
    }

    public function createGeneralCurso(Request $request)
    {
        $annio = (int) $request->query('annio', date('Y'));

        $docentes = DB::table('dm_user')
            ->select('id', 'nombres', 'apellido1', 'apellido2')
            ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->where('is_active', 1)
            ->orderBy('nombres')
            ->orderBy('apellido1')
            ->get();

        $directorCursosMap = DB::table('sweb_salones')
            ->select('director_id', 'nombre')
            ->where('is_active', 1)
            ->whereNotNull('director_id')
            ->orderBy('nombre')
            ->get()
            ->groupBy('director_id')
            ->map(function ($rows) {
                return $rows->pluck('nombre')
                    ->map(fn ($v) => trim((string) $v))
                    ->filter(fn ($v) => $v !== '')
                    ->unique()
                    ->values()
                    ->all();
            })
            ->toArray();

        $directorCursosMap = DB::table('sweb_salones')
            ->select('director_id', 'nombre')
            ->where('is_active', 1)
            ->whereNotNull('director_id')
            ->orderBy('nombre')
            ->get()
            ->groupBy('director_id')
            ->map(function ($rows) {
                return $rows->pluck('nombre')
                    ->map(fn ($v) => trim((string) $v))
                    ->filter(fn ($v) => $v !== '')
                    ->unique()
                    ->values()
                    ->all();
            })
            ->toArray();

        $directorCursosMap = DB::table('sweb_salones')
            ->select('director_id', 'nombre')
            ->where('is_active', 1)
            ->whereNotNull('director_id')
            ->orderBy('nombre')
            ->get()
            ->groupBy('director_id')
            ->map(function ($rows) {
                return $rows->pluck('nombre')
                    ->map(fn ($v) => trim((string) $v))
                    ->filter(fn ($v) => $v !== '')
                    ->unique()
                    ->values()
                    ->all();
            })
            ->toArray();

        $directorCursosMap = DB::table('sweb_salones')
            ->select('director_id', 'nombre')
            ->where('is_active', 1)
            ->whereNotNull('director_id')
            ->orderBy('nombre')
            ->get()
            ->groupBy('director_id')
            ->map(function ($rows) {
                return $rows->pluck('nombre')
                    ->map(fn ($v) => trim((string) $v))
                    ->filter(fn ($v) => $v !== '')
                    ->unique()
                    ->values()
                    ->all();
            })
            ->toArray();

        $grados = DB::table('sweb_grados as g')
            ->leftJoin('sweb_secciones as s', 's.id', '=', 'g.seccion_id')
            ->where('g.is_active', 1)
            ->select('g.id', 'g.nombre', 'g.abrev', DB::raw("COALESCE(s.nombre, '') as nivel"))
            ->orderByRaw('CASE WHEN g.orden IS NULL THEN 9999 ELSE g.orden END ASC')
            ->orderBy('g.nombre')
            ->get();

        return view('admin.general.cursos.create', [
            'annio' => $annio,
            'docentes' => $docentes,
            'grados' => $grados,
        ]);
    }

    public function storeGeneralCurso(Request $request)
    {
        $annio = (int) $request->query('annio', date('Y'));

        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:50', Rule::unique('sweb_salones', 'nombre')],
            'grado_id' => ['nullable', 'integer', Rule::exists('sweb_grados', 'id')],
            'director_id' => ['nullable', 'integer', Rule::exists('dm_user', 'id')],
            'position' => ['nullable', 'integer', 'min:0', 'max:9999'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $position = isset($data['position'])
            ? (int) $data['position']
            : ((int) DB::table('sweb_salones')->max('position') + 1);

        DB::table('sweb_salones')->insert([
            'uuid' => (string) Str::uuid(),
            'is_active' => (int) ($data['is_active'] ?? 1),
            'nombre' => trim((string) $data['nombre']),
            'grado_id' => $data['grado_id'] !== null ? (int) $data['grado_id'] : null,
            'director_id' => $data['director_id'] !== null ? (int) $data['director_id'] : 0,
            'codirector_id' => 0,
            'tot_estudiantes' => 0,
            'print_state1' => 'En CalificaciÃ³n',
            'print_state2' => 'En CalificaciÃ³n',
            'print_state3' => 'En CalificaciÃ³n',
            'print_state4' => 'En CalificaciÃ³n',
            'print_state5' => 'En CalificaciÃ³n',
            'is_ready_print' => 0,
            'print_state' => 'En CalificaciÃ³n',
            'position' => $position,
            'created_by' => 1,
            'updated_by' => 1,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        $id = (int) DB::getPdo()->lastInsertId();

        return redirect()
            ->route('admin.general.cursos.show', ['id' => $id, 'annio' => $annio])
            ->with('status', 'Curso creado correctamente.');
    }

    public function updateGeneralCursoDetalle(Request $request, int $id)
    {
        $annio = (int) $request->query('annio', date('Y'));

        $course = DB::table('sweb_salones')->where('id', $id)->first();
        if (! $course) {
            return redirect()
                ->route('admin.general.cursos.item', ['item' => 'todos', 'annio' => $annio])
                ->with('status', 'No se encontro el curso.');
        }

        $data = $request->validate([
            'director_id' => ['nullable', 'integer', Rule::exists('dm_user', 'id')],
        ]);

        DB::table('sweb_salones')
            ->where('id', $id)
            ->update([
                'director_id' => $data['director_id'] !== null ? (int) $data['director_id'] : null,
                'updated_by' => 1,
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.general.cursos.show', ['id' => $id, 'annio' => $annio])
            ->with('status', 'Curso actualizado correctamente.');
    }

    public function showGeneralAcudienteDetalle(Request $request, int $id)
    {
        $annio = (int) $request->query('annio', date('Y'));
        [$estudiantesTable, $datosTable] = $this->resolveStudentTablesByYear($annio);

        $record = DB::table("{$datosTable} as d")
            ->join("{$estudiantesTable} as e", 'e.id', '=', 'd.estudiante_id')
            ->select(
                'd.*',
                'e.id as estudiante_id',
                'e.nombres as est_nombres',
                'e.apellido1 as est_apellido1',
                'e.apellido2 as est_apellido2',
                'e.documento as est_documento',
                'e.contabilidad_id'
            )
            ->where('d.id', $id)
            ->first();

        abort_unless($record, 404);

        $keyDoc = trim((string) ($record->acudi_id ?? ''));
        $keyName = trim((string) ($record->acudiente ?? ''));

        $groupsQuery = DB::table("{$datosTable} as d")
            ->join("{$estudiantesTable} as e", 'e.id', '=', 'd.estudiante_id')
            ->select(
                'd.id',
                'd.madre', 'd.madre_id',
                'd.padre', 'd.padre_id',
                'd.acudiente', 'd.acudi_id',
                'e.id as estudiante_id',
                'e.nombres as est_nombres',
                'e.apellido1 as est_apellido1',
                'e.apellido2 as est_apellido2',
                'e.documento as est_documento',
                'e.contabilidad_id'
            )
            ->where('e.is_active', 1);

        if ($keyDoc !== '') {
            $groupsQuery->where('d.acudi_id', $keyDoc);
        } else {
            $groupsQuery->where('d.acudiente', $keyName);
        }

        $groups = $groupsQuery
            ->orderBy('e.apellido1')
            ->orderBy('e.apellido2')
            ->orderBy('e.nombres')
            ->limit(5)
            ->get()
            ->map(function ($g) {
                $estName = trim(($g->est_nombres ?? '') . ' ' . ($g->est_apellido1 ?? '') . ' ' . ($g->est_apellido2 ?? ''));
                $codigo = 'WS-' . str_pad((string) ((int) ($g->contabilidad_id ?: $g->estudiante_id)), 4, '0', STR_PAD_LEFT);
                return (object) [
                    'id' => $g->id,
                    'group_name' => $estName,
                    'members' => [
                        [
                            'nombre' => (string) ($g->madre ?? ''),
                            'parentesco' => 'Madre',
                            'documento' => (string) ($g->madre_id ?? ''),
                            'codigo' => 'N/A',
                        ],
                        [
                            'nombre' => (string) ($g->padre ?? ''),
                            'parentesco' => 'Padre',
                            'documento' => (string) ($g->padre_id ?? ''),
                            'codigo' => 'N/A',
                        ],
                        [
                            'nombre' => $estName,
                            'parentesco' => 'Estudiante',
                            'documento' => (string) ($g->est_documento ?? ''),
                            'codigo' => $codigo,
                        ],
                    ],
                ];
            });

        $acudienteFullName = trim((string) ($record->acudiente ?? ''));
        $nameParts = preg_split('/\s+/', $acudienteFullName);
        $primerNombre = $nameParts[0] ?? '';
        $segundoNombre = $nameParts[1] ?? '';
        $primerApellido = $nameParts[2] ?? '';
        $segundoApellido = implode(' ', array_slice($nameParts, 3));

        return view('admin.general.grupos.acudientes_show', [
            'annio' => $annio,
            'record' => $record,
            'groups' => $groups,
            'primerNombre' => $primerNombre,
            'segundoNombre' => $segundoNombre,
            'primerApellido' => $primerApellido,
            'segundoApellido' => $segundoApellido,
        ]);
    }

    public function updateGeneralAcudienteDetalle(Request $request, int $id)
    {
        $annio = (int) $request->query('annio', date('Y'));
        [$estudiantesTable, $datosTable] = $this->resolveStudentTablesByYear($annio);

        $row = DB::table("{$datosTable}")->where('id', $id)->first();
        if (! $row) {
            return redirect()
                ->route('admin.general.grupos.item', ['item' => 'acudientes', 'annio' => $annio])
                ->with('status', 'No se encontro el acudiente.');
        }

        $data = $request->validate([
            'primer_nombre' => ['nullable', 'string', 'max:80'],
            'segundo_nombre' => ['nullable', 'string', 'max:80'],
            'primer_apellido' => ['nullable', 'string', 'max:80'],
            'segundo_apellido' => ['nullable', 'string', 'max:80'],
            'documento' => ['nullable', 'string', 'max:30'],
            'telefono' => ['nullable', 'string', 'max:60'],
            'email' => ['nullable', 'email', 'max:120'],
            'direccion' => ['nullable', 'string', 'max:255'],
        ]);

        $acudienteNombre = trim(implode(' ', array_filter([
            $data['primer_nombre'] ?? '',
            $data['segundo_nombre'] ?? '',
            $data['primer_apellido'] ?? '',
            $data['segundo_apellido'] ?? '',
        ])));

        DB::table("{$datosTable}")
            ->where('id', $id)
            ->update([
                'acudiente' => $acudienteNombre,
                'acudi_id' => trim((string) ($data['documento'] ?? '')),
                'acudi_tel_1' => trim((string) ($data['telefono'] ?? '')),
                'acudi_email' => trim((string) ($data['email'] ?? '')),
                'acudi_dir' => trim((string) ($data['direccion'] ?? '')),
                'updated_by' => 1,
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.general.acudientes.show', ['id' => $id, 'annio' => $annio])
            ->with('status', 'Acudiente actualizado correctamente.');
    }

    public function printListaCurso(Request $request, string $tipo)
    {
        abort_unless(in_array($tipo, ['calificaciones', 'asistencia'], true), 404);

        $annio = (int) $request->query('annio', date('Y'));
        $papel = (string) $request->query('papel', 'Carta');
        $periodo = (string) $request->query('periodo', '');
        $asignatura = (string) $request->query('asignatura', '');
        $profesor = (string) $request->query('profesor', '');
        $cursoIds = collect((array) $request->query('curso_ids', []))
            ->map(fn ($id) => (int) $id)
            ->filter(fn ($id) => $id > 0)
            ->unique()
            ->values();

        if ($cursoIds->isEmpty()) {
            return redirect()
                ->route('admin.general.cursos.item', ['item' => 'listas-curso', 'annio' => $annio])
                ->withErrors(['Debe seleccionar al menos un curso de la lista']);
        }

        $cursoId = (int) $cursoIds->first();
        $curso = DB::table('sweb_salones as s')
            ->leftJoin('sweb_grados as g', 'g.id', '=', 's.grado_id')
            ->leftJoin('sweb_secciones as sec', 'sec.id', '=', 'g.seccion_id')
            ->leftJoin('dm_user as d', 'd.id', '=', 's.director_id')
            ->select(
                's.id',
                's.nombre',
                DB::raw("COALESCE(sec.nombre, '') as nivel"),
                DB::raw("TRIM(CONCAT(COALESCE(d.nombres,''),' ',COALESCE(d.apellido1,''),' ',COALESCE(d.apellido2,''))) as director")
            )
            ->where('s.id', $cursoId)
            ->first();

        if (! $curso) {
            return redirect()
                ->route('admin.general.cursos.item', ['item' => 'listas-curso', 'annio' => $annio])
                ->withErrors(['El curso seleccionado no existe.']);
        }

        $students = DB::table('sweb_estudiantes as e')
            ->select(
                'e.id',
                'e.nombres',
                'e.apellido1',
                'e.apellido2',
                'e.documento',
                DB::raw("COALESCE(NULLIF(e.contabilidad_id,0), e.id) as codigo_raw")
            )
            ->where('e.salon_id', $cursoId)
            ->where('e.is_active', 1)
            ->orderBy('e.apellido1')
            ->orderBy('e.apellido2')
            ->orderBy('e.nombres')
            ->get();

        $promedios = DB::table('sweb_notas_v2')
            ->select('estudiante_id', DB::raw('AVG(definitiva) as promedio'))
            ->where('annio', $annio)
            ->whereIn('estudiante_id', $students->pluck('id'))
            ->when($periodo !== '', fn ($q) => $q->where('periodo_id', (int) $periodo))
            ->when($asignatura !== '', fn ($q) => $q->where('asignatura_id', (int) $asignatura))
            ->groupBy('estudiante_id')
            ->pluck('promedio', 'estudiante_id');

        $students = $students->map(function ($s) use ($promedios) {
            $nota = $promedios[(int) $s->id] ?? null;
            return (object) [
                'id' => (int) $s->id,
                'codigo' => 'WS-' . str_pad((string) ((int) $s->codigo_raw), 4, '0', STR_PAD_LEFT),
                'nombre' => trim(($s->nombres ?? '') . ' ' . ($s->apellido1 ?? '') . ' ' . ($s->apellido2 ?? '')),
                'documento' => (string) ($s->documento ?? ''),
                'nota' => $nota !== null ? number_format((float) $nota, 2) : '',
            ];
        });

        $periodoLabel = '';
        if ($periodo !== '') {
            $periodoValue = DB::table('sweb_periodos')->where('id', (int) $periodo)->value('periodo');
            $periodoLabel = (string) ($periodoValue ?? '');
        }

        $asignaturaLabel = '';
        if ($asignatura !== '') {
            $asignaturaValue = DB::table('sweb_asignaturas')->where('id', (int) $asignatura)->value('nombre');
            $asignaturaLabel = (string) ($asignaturaValue ?? '');
        }

        $profesorLabel = 'Todos';
        if ($profesor !== '') {
            $profData = DB::table('dm_user')
                ->select('nombres', 'apellido1', 'apellido2')
                ->where('id', (int) $profesor)
                ->first();
            if ($profData) {
                $profesorLabel = trim(($profData->nombres ?? '') . ' ' . ($profData->apellido1 ?? '') . ' ' . ($profData->apellido2 ?? ''));
            }
        }

        $teacherSheets = collect();
        if (in_array($tipo, ['calificaciones', 'asistencia'], true)) {
            $assignTable = $this->resolveSalonAsignatProfesorTable($annio);
            $teacherSheets = DB::table("{$assignTable} as sap")
                ->join('dm_user as u', 'u.id', '=', 'sap.user_id')
                ->join('sweb_asignaturas as a', 'a.id', '=', 'sap.asignatura_id')
                ->where('sap.salon_id', $cursoId)
                ->when($profesor !== '', fn ($q) => $q->where('sap.user_id', (int) $profesor))
                ->when($asignatura !== '', fn ($q) => $q->where('sap.asignatura_id', (int) $asignatura))
                ->select(
                    'sap.user_id',
                    'sap.asignatura_id',
                    DB::raw("TRIM(CONCAT(COALESCE(u.nombres,''),' ',COALESCE(u.apellido1,''),' ',COALESCE(u.apellido2,''))) as profesor"),
                    DB::raw("UPPER(COALESCE(a.nombre,'')) as asignatura")
                )
                ->distinct()
                ->orderBy('profesor')
                ->orderBy('asignatura')
                ->get();

            if ($teacherSheets->isEmpty()) {
                $teacherSheets = collect([
                    (object) [
                        'profesor' => $profesorLabel ?: 'Todos',
                        'asignatura' => $asignaturaLabel !== '' ? mb_strtoupper($asignaturaLabel) : 'TODAS',
                    ],
                ]);
            }
        }

        return response()->view('admin.general.cursos.print', [
            'tipo' => $tipo,
            'annio' => $annio,
            'papel' => in_array($papel, ['Carta', 'Oficio'], true) ? $papel : 'Carta',
            'curso' => $curso,
            'students' => $students,
            'periodoLabel' => $periodoLabel,
            'asignaturaLabel' => $asignaturaLabel,
            'profesorLabel' => $profesorLabel,
            'teacherSheets' => $teacherSheets,
        ]);
    }

    public function showAcademiaItem(string $item)
    {
        $allowed = [
            'asignaturas',
            'areas-academicas',
            'periodos-academicos',
        ];

        abort_unless(in_array($item, $allowed, true), 404);

        if ($item === 'periodos-academicos') {
            return $this->renderPeriodosAcademicos();
        }
        if ($item === 'areas-academicas') {
            return $this->renderAreasAcademicas();
        }
        if ($item === 'asignaturas') {
            return $this->renderAsignaturas();
        }

        $titles = [
            'asignaturas' => 'Academia - Asignaturas',
            'areas-academicas' => 'Academia - Areas Academicas',
            'periodos-academicos' => 'Academia - Periodos Academicos',
        ];

        return view('admin.section', [
            'title' => $titles[$item] ?? Str::title(str_replace('-', ' ', $item)),
            'section' => 'academia.' . $item,
        ]);
    }

    public function showAcademiaAsignacionItem(string $item)
    {
        $allowed = [
            'estado-actual',
            'por-curso',
            'por-profesor-asignatura',
        ];

        abort_unless(in_array($item, $allowed, true), 404);

        if ($item === 'por-profesor-asignatura') {
            return $this->renderAsignacionPorProfesorAsignatura();
        }
        if ($item === 'por-curso') {
            return $this->renderAsignacionPorCurso();
        }
        if ($item === 'estado-actual') {
            return $this->renderAsignacionEstadoActual();
        }

        $titles = [
            'estado-actual' => 'Academia - Asignacion Academica - Estado Actual',
            'por-curso' => 'Academia - Asignacion Academica - Por Curso',
            'por-profesor-asignatura' => 'Academia - Asignacion Academica - Por Profesor y Asignatura',
        ];

        return view('admin.section', [
            'title' => $titles[$item] ?? Str::title(str_replace('-', ' ', $item)),
            'section' => 'academia.asignacion-academica.' . $item,
        ]);
    }

    public function showCalificacionesItem(string $item)
    {
        $allowed = [
            'niveles-calificaciones',
            'logros',
            'periodos-notas',
            'curso-a-cargo',
            'calificaciones-faltantes',
            'registro-calificaciones',
            'generacion-boletines',
            'reportes',
        ];

        abort_unless(in_array($item, $allowed, true), 404);

        if ($item === 'niveles-calificaciones') {
            return $this->renderNivelesCalificaciones();
        }
        if ($item === 'logros') {
            return $this->renderLogrosCalificaciones();
        }
        if ($item === 'periodos-notas') {
            $role = Str::lower((string) session('admin_role', ''));
            abort_unless(in_array($role, ['admin', 'superadmin', 'coordinadores'], true), 403);
            return $this->renderPeriodosNotas();
        }
        if ($item === 'curso-a-cargo') {
            return $this->renderCursoACargo();
        }

        $titles = [
            'niveles-calificaciones' => 'Calificaciones - Niveles de Calificaciones',
            'logros' => 'Calificaciones - Logros',
            'periodos-notas' => 'Calificaciones - Apertura de Notas',
            'curso-a-cargo' => 'Calificaciones - Curso a cargo',
            'calificaciones-faltantes' => 'Calificaciones - Calificaciones Faltantes',
            'registro-calificaciones' => 'Calificaciones - Registro de Calificaciones',
            'generacion-boletines' => 'Calificaciones - Generacion de Boletines',
            'reportes' => 'Calificaciones - Reportes',
        ];

        return view('admin.section', [
            'title' => $titles[$item] ?? Str::title(str_replace('-', ' ', $item)),
            'section' => 'calificaciones.' . $item,
        ]);
    }

    private function renderNivelesCalificaciones()
    {
        $rows = DB::table('sweb_rangos')
            ->select('id', 'nombre', 'limite_inferior', 'limite_superior', 'orden')
            ->orderBy('orden')
            ->orderBy('id')
            ->get()
            ->map(function ($row) {
                return (object) [
                    'id' => (int) $row->id,
                    'nombre' => trim((string) ($row->nombre ?? '')),
                    'rango' => ((int) $row->limite_inferior) . ' - ' . ((int) $row->limite_superior),
                    'editable' => true,
                ];
            });

        return view('admin.calificaciones.niveles', [
            'rows' => $rows->values(),
        ]);
    }

    private function renderPeriodosNotas()
    {
        $annio = (int) request('annio', date('Y'));
        $today = Carbon::today()->toDateString();

        $rows = DB::table('sweb_periodos')
            ->select('id', 'periodo', 'label', 'f_ini_notas', 'f_fin_notas')
            ->orderBy('id')
            ->get()
            ->map(function ($row) use ($annio, $today) {
                $periodoTexto = trim((string) ($row->periodo ?? ''));
                $numero = $this->extractPeriodNumber($periodoTexto);
                $label = trim((string) ($row->label ?? ''));

                if ($label === '') {
                    $label = $numero !== null ? ($numero . ' - ' . $annio) : $periodoTexto;
                } else {
                    // Si la etiqueta tiene año, lo alinea con el año académico seleccionado.
                    $label = preg_replace('/\b(19|20)\d{2}\b/u', (string) $annio, $label);
                }

                $fechaIni = $row->f_ini_notas ? Carbon::parse($row->f_ini_notas)->toDateString() : '';
                $fechaFin = $row->f_fin_notas ? Carbon::parse($row->f_fin_notas)->toDateString() : '';

                $habilitado = $fechaIni !== '' && $fechaFin !== ''
                    && $today >= $fechaIni
                    && $today <= $fechaFin;

                return (object) [
                    'id' => (int) $row->id,
                    'periodo' => $periodoTexto,
                    'label' => $label,
                    'f_ini_notas' => $fechaIni,
                    'f_fin_notas' => $fechaFin,
                    'habilitado' => $habilitado,
                ];
            });

        return view('admin.calificaciones.periodos_notas', [
            'annio' => $annio,
            'rows' => $rows,
        ]);
    }

    private function renderCursoACargo()
    {
        $annio = (int) request('annio', date('Y'));
        $userId = (int) session('admin_user_id', 0);
        $role = Str::lower((string) session('admin_role', ''));
        $isAdminView = in_array($role, ['admin', 'superadmin', 'coordinadores'], true);
        $selectedDocente = (int) request('profesor', 0);

        $docentes = DB::table('dm_user')
            ->select('id', 'nombres', 'apellido1', 'apellido2')
            ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->where('is_active', 1)
            ->orderBy('nombres')
            ->orderBy('apellido1')
            ->get();

        $directorCursosMap = DB::table('sweb_salones')
            ->select('director_id', 'nombre')
            ->where('is_active', 1)
            ->whereNotNull('director_id')
            ->orderBy('nombre')
            ->get()
            ->groupBy('director_id')
            ->map(function ($rows) {
                return $rows->pluck('nombre')
                    ->map(fn ($v) => trim((string) $v))
                    ->filter(fn ($v) => $v !== '')
                    ->unique()
                    ->values()
                    ->all();
            })
            ->toArray();

        if (! $isAdminView) {
            $selectedDocente = $userId;
        } elseif ($role === 'coordinadores' && $selectedDocente === 0 && $userId > 0) {
            $existsInList = $docentes->firstWhere('id', $userId);
            if ($existsInList) {
                $selectedDocente = $userId;
            }
        }

        $docenteNombre = '';
        if ($selectedDocente > 0) {
            $selected = $docentes->firstWhere('id', $selectedDocente);
            if ($selected) {
                $docenteNombre = trim(($selected->nombres ?? '') . ' ' . ($selected->apellido1 ?? '') . ' ' . ($selected->apellido2 ?? ''));
            }
        }

        $courses = DB::table('sweb_salones')
            ->select('id', 'nombre', 'grado_id')
            ->where('is_active', 1)
            ->when($selectedDocente > 0, fn ($q) => $q->where('director_id', $selectedDocente), fn ($q) => $q->whereRaw('1=0'))
            ->orderBy('position')
            ->orderBy('nombre')
            ->get();

        if ($courses->isEmpty()) {
            return view('admin.calificaciones.curso_a_cargo', [
                'annio' => $annio,
                'courses' => collect(),
                'periodos' => collect(),
                'isAdminView' => $isAdminView,
                'docentes' => $docentes,
                'directorCursosMap' => $directorCursosMap,
                'selectedDocente' => $selectedDocente,
                'docenteNombre' => $docenteNombre,
            ]);
        }

        [$studentsTable, $datosTable] = $this->resolveStudentTablesByYear($annio);
        $salonIds = $courses->pluck('id')->map(fn ($v) => (int) $v)->values();
        $gradoIds = $courses->pluck('grado_id')->map(fn ($v) => (int) $v)->unique()->values();

        $students = DB::table("{$studentsTable} as e")
            ->leftJoin("{$datosTable} as d", 'd.estudiante_id', '=', 'e.id')
            ->select(
                'e.id',
                'e.salon_id',
                'e.nombres',
                'e.apellido1',
                'e.apellido2',
                'e.contabilidad_id',
                'd.madre',
                'd.madre_id',
                'd.padre',
                'd.padre_id'
            )
            ->whereIn('e.salon_id', $salonIds)
            ->where('e.is_active', 1)
            ->orderBy('e.apellido1')
            ->orderBy('e.apellido2')
            ->orderBy('e.nombres')
            ->get();

        $studentIds = $students->pluck('id')->map(fn ($v) => (int) $v)->values();

        $periodos = DB::table('sweb_periodos')
            ->select('id', 'periodo', 'label')
            ->orderBy('id')
            ->get()
            ->map(function ($p) use ($annio) {
                $periodoTxt = trim((string) ($p->periodo ?? ''));
                $numero = $this->extractPeriodNumber($periodoTxt);
                $label = $numero !== null ? ($numero . '° - ' . $annio) : $periodoTxt;
                return (object) [
                    'id' => (int) $p->id,
                    'label' => $label !== '' ? $label : ('Periodo ' . (int) $p->id . ' - ' . $annio),
                ];
            });

        $subjectRows = DB::table('sweb_grado_asignat as ga')
            ->join('sweb_asignaturas as a', 'a.id', '=', 'ga.asignatura_id')
            ->select('ga.grado_id', 'a.id as asignatura_id', 'a.nombre', 'a.abrev', 'ga.orden')
            ->whereIn('ga.grado_id', $gradoIds)
            ->where('a.is_active', 1)
            ->orderByRaw('CASE WHEN ga.orden IS NULL THEN 9999 ELSE ga.orden END ASC')
            ->orderBy('a.nombre')
            ->get();

        $subjectsByGrade = [];
        foreach ($subjectRows as $s) {
            $g = (int) $s->grado_id;
            if (! isset($subjectsByGrade[$g])) {
                $subjectsByGrade[$g] = [];
            }
            $subjectsByGrade[$g][] = (object) [
                'id' => (int) $s->asignatura_id,
                'abrev' => trim((string) ($s->abrev ?: $s->nombre)),
                'nombre' => trim((string) ($s->nombre ?? '')),
            ];
        }

        $notasMap = [];
        $promMap = [];
        if ($studentIds->isNotEmpty()) {
            $notas = DB::table('sweb_notas_v2')
                ->select('estudiante_id', 'periodo_id', 'asignatura_id', 'definitiva')
                ->where('annio', $annio)
                ->whereIn('estudiante_id', $studentIds)
                ->get();

            foreach ($notas as $n) {
                $k = (int) $n->estudiante_id . '|' . (int) $n->periodo_id . '|' . (int) $n->asignatura_id;
                $notasMap[$k] = $n->definitiva !== null ? number_format((float) $n->definitiva, 1) : '';
            }

            $promRows = DB::table('sweb_notas_v2')
                ->select('estudiante_id', 'periodo_id', DB::raw('AVG(definitiva) as prom'))
                ->where('annio', $annio)
                ->whereIn('estudiante_id', $studentIds)
                ->groupBy('estudiante_id', 'periodo_id')
                ->get();

            foreach ($promRows as $r) {
                $k = (int) $r->estudiante_id . '|' . (int) $r->periodo_id;
                $promMap[$k] = $r->prom !== null ? number_format((float) $r->prom, 1) : '';
            }
        }

        $studentsBySalon = $students->groupBy('salon_id');

        $courses = $courses->map(function ($course) use ($subjectsByGrade, $studentsBySalon, $periodos, $notasMap, $promMap) {
            $subjects = collect($subjectsByGrade[(int) $course->grado_id] ?? [])->values();
            $rawStudents = collect($studentsBySalon->get((int) $course->id, []))->values();

            $mappedStudents = $rawStudents->map(function ($st) use ($subjects, $periodos, $notasMap, $promMap) {
                $nombre = trim(($st->apellido1 ?? '') . ' ' . ($st->apellido2 ?? '') . ', ' . ($st->nombres ?? ''));
                $codigo = (int) ($st->contabilidad_id ?: $st->id);

                $padres = collect([
                    trim((string) ($st->madre ?? '')) !== '' ? trim((string) $st->madre) . (trim((string) ($st->madre_id ?? '')) !== '' ? ' [' . trim((string) $st->madre_id) . ']' : '') : null,
                    trim((string) ($st->padre ?? '')) !== '' ? trim((string) $st->padre) . (trim((string) ($st->padre_id ?? '')) !== '' ? ' [' . trim((string) $st->padre_id) . ']' : '') : null,
                ])->filter()->values()->join(' / ');

                $periodRows = $periodos->map(function ($p) use ($st, $subjects, $notasMap, $promMap) {
                    $prom = $promMap[(int) $st->id . '|' . (int) $p->id] ?? '';
                    $subjectValues = $subjects->map(function ($subj) use ($st, $p, $notasMap) {
                        $key = (int) $st->id . '|' . (int) $p->id . '|' . (int) $subj->id;
                        return $notasMap[$key] ?? '';
                    });
                    return (object) [
                        'periodo_id' => (int) $p->id,
                        'periodo' => $p->label,
                        'prom' => $prom,
                        'values' => $subjectValues,
                    ];
                });

                return (object) [
                    'id' => (int) $st->id,
                    'codigo' => $codigo,
                    'nombre' => $nombre !== '' ? $nombre : 'Sin nombre',
                    'padres' => $padres,
                    'rows' => $periodRows,
                ];
            });

            $course->subjects = $subjects;
            $course->students = $mappedStudents;
            return $course;
        });

        return view('admin.calificaciones.curso_a_cargo', [
            'annio' => $annio,
            'courses' => $courses,
            'periodos' => $periodos,
            'isAdminView' => $isAdminView,
            'docentes' => $docentes,
            'directorCursosMap' => $directorCursosMap,
            'selectedDocente' => $selectedDocente,
            'docenteNombre' => $docenteNombre,
        ]);
    }

    public function savePeriodosNotas(Request $request)
    {
        $role = Str::lower((string) session('admin_role', ''));
        abort_unless(in_array($role, ['admin', 'superadmin', 'coordinadores'], true), 403);

        $annio = (int) $request->input('annio', date('Y'));
        $periodos = $request->input('periodos', []);
        if (! is_array($periodos)) {
            $periodos = [];
        }

        $rows = DB::table('sweb_periodos')
            ->select('id')
            ->whereIn('id', array_map('intval', array_keys($periodos)))
            ->get()
            ->keyBy('id');

        foreach ($periodos as $id => $config) {
            $id = (int) $id;
            if (! isset($rows[$id])) {
                continue;
            }

            $ini = trim((string) data_get($config, 'f_ini_notas', ''));
            $fin = trim((string) data_get($config, 'f_fin_notas', ''));

            try {
                $iniDate = $ini !== '' ? Carbon::createFromFormat('Y-m-d', $ini) : null;
                $finDate = $fin !== '' ? Carbon::createFromFormat('Y-m-d', $fin) : null;
            } catch (\Throwable $e) {
                return redirect()
                    ->route('admin.calificaciones.item', ['item' => 'periodos-notas', 'annio' => $annio])
                    ->withErrors(['general' => 'Formato de fecha no valido. Use YYYY-MM-DD.']);
            }

            if (($iniDate && ! $finDate) || (! $iniDate && $finDate)) {
                return redirect()
                    ->route('admin.calificaciones.item', ['item' => 'periodos-notas', 'annio' => $annio])
                    ->withErrors(['general' => 'Cada periodo debe tener Fecha Inicio y Fecha Fin de notas.']);
            }

            if ($iniDate && $finDate && $finDate->lt($iniDate)) {
                return redirect()
                    ->route('admin.calificaciones.item', ['item' => 'periodos-notas', 'annio' => $annio])
                    ->withErrors(['general' => 'La Fecha Fin de notas no puede ser menor que la Fecha Inicio.']);
            }

            DB::table('sweb_periodos')
                ->where('id', $id)
                ->update([
                    'f_ini_notas' => $iniDate?->toDateString(),
                    'f_fin_notas' => $finDate?->toDateString(),
                    'updated_by' => 1,
                    'updated_at' => now(),
                ]);
        }

        return redirect()
            ->route('admin.calificaciones.item', ['item' => 'periodos-notas', 'annio' => $annio])
            ->with('status', 'Rangos de notas guardados correctamente.');
    }

    public function togglePeriodoNotas(Request $request, int $id)
    {
        $role = Str::lower((string) session('admin_role', ''));
        abort_unless(in_array($role, ['admin', 'superadmin', 'coordinadores'], true), 403);

        $annio = (int) $request->input('annio', date('Y'));
        $row = DB::table('sweb_periodos')
            ->select('id', 'f_ini_notas', 'f_fin_notas')
            ->where('id', $id)
            ->first();

        if (! $row) {
            return redirect()
                ->route('admin.calificaciones.item', ['item' => 'periodos-notas', 'annio' => $annio])
                ->withErrors(['general' => 'No se encontro el periodo.']);
        }

        $today = Carbon::today();
        $ini = $row->f_ini_notas ? Carbon::parse($row->f_ini_notas) : null;
        $fin = $row->f_fin_notas ? Carbon::parse($row->f_fin_notas) : null;

        $isEnabled = $ini && $fin
            && $today->betweenIncluded($ini->copy()->startOfDay(), $fin->copy()->startOfDay());

        if ($isEnabled) {
            DB::table('sweb_periodos')
                ->where('id', $id)
                ->update([
                    'f_ini_notas' => null,
                    'f_fin_notas' => null,
                    'updated_by' => 1,
                    'updated_at' => now(),
                ]);

            return redirect()
                ->route('admin.calificaciones.item', ['item' => 'periodos-notas', 'annio' => $annio])
                ->with('status', 'Periodo deshabilitado para carga de notas.');
        }

        $newIni = $ini ? $ini->copy() : $today->copy();
        $newFin = $fin ? $fin->copy() : $today->copy();

        if ($newFin->lt($newIni)) {
            $tmp = $newIni->copy();
            $newIni = $newFin->copy();
            $newFin = $tmp;
        }

        if ($today->lt($newIni)) {
            $newIni = $today->copy();
        }
        if ($today->gt($newFin)) {
            $newFin = $today->copy();
        }

        DB::table('sweb_periodos')
            ->where('id', $id)
            ->update([
                'f_ini_notas' => $newIni->toDateString(),
                'f_fin_notas' => $newFin->toDateString(),
                'updated_by' => 1,
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.calificaciones.item', ['item' => 'periodos-notas', 'annio' => $annio])
            ->with('status', 'Periodo habilitado para carga de notas.');
    }

    private function renderLogrosCalificaciones()
    {
        $annio = (int) request('annio', date('Y'));
        $estado = (string) request('estado', 'habilitados');
        $profesor = (string) request('profesor', '');
        $grado = (string) request('grado', '');
        $asignatura = (string) request('asignatura', '');
        $periodo = (string) request('periodo', '');
        $nivel = (string) request('nivel', '');

        $profesores = DB::table('dm_user')
            ->select('id', 'nombres', 'apellido1', 'apellido2')
            ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->where('is_active', 1)
            ->orderBy('nombres')
            ->orderBy('apellido1')
            ->get();

        $grados = DB::table('sweb_grados')
            ->select('id', 'nombre')
            ->where('is_active', 1)
            ->orderBy('orden')
            ->orderBy('nombre')
            ->get();

        $asignaturas = DB::table('sweb_asignaturas')
            ->select('id', 'nombre')
            ->where('is_active', 1)
            ->orderBy('nombre')
            ->get();

        $periodos = DB::table('sweb_periodos')
            ->select('id', 'periodo')
            ->orderBy('id')
            ->get();

        $rangos = DB::table('sweb_rangos')
            ->select('id', 'nombre', 'limite_inferior', 'limite_superior')
            ->orderBy('orden')
            ->orderBy('id')
            ->get();

        $query = DB::table('sweb_indicadores as i')
            ->leftJoin('sweb_grados as g', 'g.id', '=', 'i.grado_id')
            ->leftJoin('sweb_asignaturas as a', 'a.id', '=', 'i.asignatura_id')
            ->leftJoin('sweb_periodos as p', 'p.id', '=', 'i.periodo_id')
            ->leftJoin('dm_user as cu', 'cu.id', '=', 'i.created_by')
            ->leftJoin('dm_user as mu', 'mu.id', '=', 'i.updated_by')
            ->where('i.annio', $annio)
            ->where('i.is_active', 1)
            ->when($estado === 'deshabilitados', fn ($q) => $q->where('i.is_visible', 0))
            ->when($estado !== 'deshabilitados', fn ($q) => $q->where('i.is_visible', 1))
            ->when($profesor !== '', function ($q) use ($profesor) {
                $q->where(function ($w) use ($profesor) {
                    $w->where('i.created_by', (int) $profesor)
                        ->orWhere('i.updated_by', (int) $profesor);
                });
            })
            ->when($grado !== '', fn ($q) => $q->where('i.grado_id', (int) $grado))
            ->when($asignatura !== '', fn ($q) => $q->where('i.asignatura_id', (int) $asignatura))
            ->when($periodo !== '', fn ($q) => $q->where('i.periodo_id', (int) $periodo));

        if ($nivel !== '') {
            $nivelKey = Str::lower(Str::ascii($nivel));
            if ($nivelKey === 'no clasificado') {
                $query->where(function ($w) use ($rangos) {
                    $w->whereRaw("i.codigo IS NULL OR i.codigo = '' OR i.codigo NOT REGEXP '^[0-9]+$'");
                    if ($rangos->isNotEmpty()) {
                        $w->orWhere(function ($w2) use ($rangos) {
                            $w2->whereRaw("i.codigo REGEXP '^[0-9]+$'");
                            $w2->where(function ($w3) use ($rangos) {
                                foreach ($rangos as $r) {
                                    $w3->whereRaw(
                                        "CAST(i.codigo AS UNSIGNED) NOT BETWEEN ? AND ?",
                                        [(int) $r->limite_inferior, (int) $r->limite_superior]
                                    );
                                }
                            });
                        });
                    }
                });
            } else {
                $rangoSel = $rangos->first(function ($r) use ($nivelKey) {
                    return Str::lower(Str::ascii((string) $r->nombre)) === $nivelKey;
                });
                if ($rangoSel) {
                    $query->whereRaw("i.codigo REGEXP '^[0-9]+$'")
                        ->whereRaw(
                            "CAST(i.codigo AS UNSIGNED) BETWEEN ? AND ?",
                            [(int) $rangoSel->limite_inferior, (int) $rangoSel->limite_superior]
                        );
                }
            }
        }

        $rows = $query
            ->select(
                'i.id',
                'i.codigo',
                'i.concepto',
                'i.created_at',
                'i.updated_at',
                'i.created_by',
                'i.updated_by',
                'g.nombre as grado_nombre',
                'a.nombre as asignatura_nombre',
                'p.periodo as periodo_nombre',
                DB::raw("TRIM(CONCAT(COALESCE(cu.nombres,''),' ',COALESCE(cu.apellido1,''),' ',COALESCE(cu.apellido2,''))) as creado_por"),
                DB::raw("TRIM(CONCAT(COALESCE(mu.nombres,''),' ',COALESCE(mu.apellido1,''),' ',COALESCE(mu.apellido2,''))) as modificado_por")
            )
            ->orderBy('i.grado_id')
            ->orderBy('i.asignatura_id')
            ->orderBy('i.periodo_id')
            ->orderBy('i.id')
            ->paginate(20)
            ->withQueryString();

        $rows->getCollection()->transform(function ($row) use ($rangos) {
            $code = is_numeric($row->codigo) ? (int) $row->codigo : null;
            $nivelLabel = 'No Clasificado';
            if ($code !== null) {
                $match = $rangos->first(function ($r) use ($code) {
                    return $code >= (int) $r->limite_inferior && $code <= (int) $r->limite_superior;
                });
                if ($match) {
                    $nivelLabel = mb_strtoupper((string) $match->nombre);
                }
            }
            $row->nivel_calificacion = $nivelLabel;
            return $row;
        });

        return view('admin.calificaciones.logros', [
            'annio' => $annio,
            'estado' => $estado === 'deshabilitados' ? 'deshabilitados' : 'habilitados',
            'profesor' => $profesor,
            'grado' => $grado,
            'asignatura' => $asignatura,
            'periodo' => $periodo,
            'nivel' => $nivel,
            'profesores' => $profesores,
            'grados' => $grados,
            'asignaturas' => $asignaturas,
            'periodos' => $periodos,
            'rangos' => $rangos,
            'rows' => $rows,
        ]);
    }

    public function addNivelCalificacion(Request $request)
    {
        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:255'],
            'rango' => ['required', 'string', 'max:30'],
        ]);

        [$from, $to] = $this->parseRange($data['rango']);
        if ($from === null || $to === null) {
            return back()->withErrors(['rango' => 'El rango debe tener formato 1 - 59'])->withInput();
        }

        try {
            DB::table('sweb_rangos')->insert([
                'nombre' => trim($data['nombre']),
                'limite_inferior' => (int) round($from),
                'limite_superior' => (int) round($to),
                'color_rango' => '0D6F88',
                'color_texto' => 'FFFFFF',
                'color_fondo' => '0D6F88',
                'created_by' => 1,
                'updated_by' => 1,
                'created_at' => now(),
                'updated_at' => now(),
                'orden' => ((int) DB::table('sweb_rangos')->max('orden')) + 1,
            ]);
        } catch (\Throwable $e) {
            return back()->withErrors(['general' => 'No se pudo agregar el nivel. Verifica que el nombre y el rango no esten repetidos.']);
        }

        return back()->with('status', 'Nivel agregado correctamente.');
    }

    public function createNivelCalificacion()
    {
        return view('admin.calificaciones.nivel_show', [
            'nivel' => null,
        ]);
    }

    public function storeNivelCalificacion(Request $request)
    {
        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:255'],
            'desde' => ['required', 'numeric', 'min:0', 'max:100'],
            'hasta' => ['required', 'numeric', 'min:0', 'max:100'],
        ]);

        $desde = (float) $data['desde'];
        $hasta = (float) $data['hasta'];
        if ($desde > $hasta) {
            return back()->withErrors(['hasta' => 'Hasta debe ser mayor o igual a Desde.'])->withInput();
        }

        try {
            DB::table('sweb_rangos')->insert([
                'nombre' => trim($data['nombre']),
                'limite_inferior' => (int) round($desde),
                'limite_superior' => (int) round($hasta),
                'color_rango' => '0D6F88',
                'color_texto' => 'FFFFFF',
                'color_fondo' => '0D6F88',
                'created_by' => 1,
                'updated_by' => 1,
                'created_at' => now(),
                'updated_at' => now(),
                'orden' => ((int) DB::table('sweb_rangos')->max('orden')) + 1,
            ]);
        } catch (\Throwable $e) {
            return back()->withErrors(['general' => 'No se pudo guardar el nivel. Verifica nombre y rango.'])->withInput();
        }

        return redirect()->route('admin.calificaciones.item', ['item' => 'niveles-calificaciones'])->with('status', 'Nivel agregado correctamente.');
    }

    public function showNivelCalificacion(int $id)
    {
        $nivel = DB::table('sweb_rangos')
            ->select('id', 'nombre', 'limite_inferior', 'limite_superior')
            ->where('id', $id)
            ->first();

        abort_unless($nivel, 404);

        return view('admin.calificaciones.nivel_show', [
            'nivel' => $nivel,
        ]);
    }

    public function updateNivelCalificacion(Request $request, int $id)
    {
        $exists = DB::table('sweb_rangos')->where('id', $id)->exists();
        if (! $exists) {
            return redirect()->route('admin.calificaciones.item', ['item' => 'niveles-calificaciones'])->withErrors(['general' => 'No se encontro el nivel.']);
        }

        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:255'],
            'desde' => ['required', 'numeric', 'min:0', 'max:100'],
            'hasta' => ['required', 'numeric', 'min:0', 'max:100'],
        ]);

        $desde = (float) $data['desde'];
        $hasta = (float) $data['hasta'];
        if ($desde > $hasta) {
            return back()->withErrors(['hasta' => 'Hasta debe ser mayor o igual a Desde.'])->withInput();
        }

        try {
            DB::table('sweb_rangos')->where('id', $id)->update([
                'nombre' => trim($data['nombre']),
                'limite_inferior' => (int) round($desde),
                'limite_superior' => (int) round($hasta),
                'updated_by' => 1,
                'updated_at' => now(),
            ]);
        } catch (\Throwable $e) {
            return back()->withErrors(['general' => 'No se pudo guardar el nivel.'])->withInput();
        }

        return redirect()->route('admin.calificaciones.item', ['item' => 'niveles-calificaciones'])->with('status', 'Nivel actualizado correctamente.');
    }

    public function editNivelCalificacion(Request $request, int $id)
    {
        $exists = DB::table('sweb_rangos')->where('id', $id)->exists();
        if (! $exists) {
            return back()->withErrors(['general' => 'No se encontro el nivel seleccionado.']);
        }

        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:255'],
            'rango' => ['required', 'string', 'max:30'],
        ]);

        [$from, $to] = $this->parseRange($data['rango']);
        if ($from === null || $to === null) {
            return back()->withErrors(['rango' => 'El rango debe tener formato 1 - 59'])->withInput();
        }

        try {
            DB::table('sweb_rangos')
                ->where('id', $id)
                ->update([
                    'nombre' => trim($data['nombre']),
                    'limite_inferior' => (int) round($from),
                    'limite_superior' => (int) round($to),
                    'updated_by' => 1,
                    'updated_at' => now(),
                ]);
        } catch (\Throwable $e) {
            return back()->withErrors(['general' => 'No se pudo editar el nivel. Verifica que el nombre y el rango no esten repetidos.']);
        }

        return back()->with('status', 'Nivel actualizado correctamente.');
    }

    public function deleteNivelCalificacion(Request $request, int $id)
    {
        $exists = DB::table('sweb_rangos')->where('id', $id)->exists();
        if (! $exists) {
            return back()->withErrors(['general' => 'No se encontro el nivel seleccionado.']);
        }

        try {
            DB::table('sweb_rangos')->where('id', $id)->delete();
        } catch (\Throwable $e) {
            return back()->withErrors(['general' => 'No se pudo eliminar el nivel porque esta en uso.']);
        }

        return back()->with('status', 'Nivel eliminado correctamente.');
    }

    private function parseRange(string $input): array
    {
        $text = trim(str_replace(',', '.', $input));
        if (! preg_match('/^\s*(\d{1,3}(?:\.\d+)?)\s*-\s*(\d{1,3}(?:\.\d+)?)\s*$/', $text, $m)) {
            return [null, null];
        }

        $from = (float) $m[1];
        $to = (float) $m[2];
        if ($from < 0 || $to < 0 || $from > 100 || $to > 100 || $from > $to) {
            return [null, null];
        }

        return [$from, $to];
    }

    private function renderAsignacionPorProfesorAsignatura()
    {
        $annio = (int) request('annio', date('Y'));
        $profesor = (string) request('profesor', '');
        $asignatura = (string) request('asignatura', '');

        $profesores = DB::table('dm_user')
            ->select('id', 'nombres', 'apellido1', 'apellido2')
            ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->where('is_active', 1)
            ->orderBy('nombres')
            ->orderBy('apellido1')
            ->get();

        $asignaturas = DB::table('sweb_asignaturas')
            ->select('id', 'nombre')
            ->where('is_active', 1)
            ->orderBy('nombre')
            ->get();

        $courses = collect();
        $selectedSalonIds = [];
        if ($profesor !== '' && $asignatura !== '') {
            $gradoIds = DB::table('sweb_grado_asignat')
                ->where('asignatura_id', (int) $asignatura)
                ->pluck('grado_id')
                ->unique()
                ->values();

            $courses = DB::table('sweb_salones')
                ->select('id', 'nombre', 'grado_id', 'position')
                ->whereIn('grado_id', $gradoIds)
                ->orderBy('grado_id')
                ->orderBy('position')
                ->orderBy('nombre')
                ->get();

            $assignTable = $this->resolveSalonAsignatProfesorTable($annio);
            $selectedSalonIds = DB::table($assignTable)
                ->where('user_id', (int) $profesor)
                ->where('asignatura_id', (int) $asignatura)
                ->pluck('salon_id')
                ->map(fn ($v) => (int) $v)
                ->all();
        }

        return view('admin.academia.asignacion_profesor_asignatura', [
            'title' => 'Asignacion Academica por Profesor y Asignatura',
            'annio' => $annio,
            'profesores' => $profesores,
            'asignaturas' => $asignaturas,
            'profesor' => $profesor,
            'asignatura' => $asignatura,
            'courses' => $courses,
            'selectedSalonIds' => $selectedSalonIds,
        ]);
    }

    private function renderAsignacionPorCurso()
    {
        $annio = (int) request('annio', date('Y'));
        $curso = (string) request('curso', '');

        $courses = DB::table('sweb_salones as s')
            ->leftJoin('sweb_grados as g', 'g.id', '=', 's.grado_id')
            ->select('s.id', 's.nombre', 's.grado_id', 'g.nombre as grado_nombre', 's.position')
            ->where('s.is_active', 1)
            ->orderBy('s.grado_id')
            ->orderBy('s.position')
            ->orderBy('s.nombre')
            ->get();

        $profesores = DB::table('dm_user')
            ->select('id', 'nombres', 'apellido1', 'apellido2')
            ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->where('is_active', 1)
            ->orderBy('nombres')
            ->orderBy('apellido1')
            ->get();

        $subjects = collect();
        $selectedBySubject = [];
        if ($curso !== '') {
            $salon = DB::table('sweb_salones')->select('id', 'grado_id')->where('id', (int) $curso)->first();
            if ($salon) {
                $subjects = DB::table('sweb_grado_asignat as ga')
                    ->join('sweb_asignaturas as a', 'a.id', '=', 'ga.asignatura_id')
                    ->leftJoin('sweb_areas as ar', 'ar.id', '=', 'a.area_id')
                    ->select('a.id', 'a.nombre', 'ar.nombre as area_nombre', 'ga.orden')
                    ->where('ga.grado_id', (int) $salon->grado_id)
                    ->where('a.is_active', 1)
                    ->orderByRaw('CASE WHEN ga.orden IS NULL THEN 9999 ELSE ga.orden END ASC')
                    ->orderBy('a.nombre')
                    ->get();

                $assignTable = $this->resolveSalonAsignatProfesorTable($annio);
                $selectedBySubject = DB::table($assignTable)
                    ->select('asignatura_id', DB::raw('SUBSTRING_INDEX(GROUP_CONCAT(user_id ORDER BY id DESC), ",", 1) as user_id'))
                    ->where('salon_id', (int) $curso)
                    ->groupBy('asignatura_id')
                    ->pluck('user_id', 'asignatura_id')
                    ->map(fn ($v) => (int) $v)
                    ->all();
            }
        }

        return view('admin.academia.asignacion_curso', [
            'title' => 'Asignacion Academica por Curso',
            'annio' => $annio,
            'curso' => $curso,
            'courses' => $courses,
            'profesores' => $profesores,
            'subjects' => $subjects,
            'selectedBySubject' => $selectedBySubject,
        ]);
    }

    private function renderAsignacionEstadoActual()
    {
        $annio = (int) request('annio', date('Y'));
        $profesor = (string) request('profesor', '');
        $cursoId = (int) request('curso_id', 0);
        $asignaturaId = (int) request('asignatura_id', 0);
        $periodoId = (int) request('periodo_id', 0);
        $role = Str::lower((string) session('admin_role', ''));
        $sessionUserId = (int) session('admin_user_id', 0);
        $isDocenteView = $role === 'docentes';

        $profesores = DB::table('dm_user')
            ->select('id', 'nombres', 'apellido1', 'apellido2')
            ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
            ->where('is_active', 1)
            ->orderBy('nombres')
            ->orderBy('apellido1')
            ->get();

        if ($isDocenteView) {
            $profesor = $sessionUserId > 0 ? (string) $sessionUserId : '';
            $profesores = $profesores->where('id', $sessionUserId)->values();
        } elseif ($role === 'coordinadores' && $profesor === '' && $sessionUserId > 0) {
            $existsInList = $profesores->firstWhere('id', $sessionUserId);
            if ($existsInList) {
                $profesor = (string) $sessionUserId;
            }
        }

        $docenteNombre = '';
        if ($profesor !== '') {
            $selected = $profesores->firstWhere('id', (int) $profesor);
            if ($selected) {
                $docenteNombre = trim(($selected->nombres ?? '') . ' ' . ($selected->apellido1 ?? '') . ' ' . ($selected->apellido2 ?? ''));
            }
        }

        $periodosIndicadores = DB::table('sweb_periodos')
            ->select('id', 'periodo')
            ->orderBy('id')
            ->get()
            ->map(function ($p) {
                $num = $this->extractPeriodNumber((string) ($p->periodo ?? ''));
                return (object) [
                    'id' => (int) $p->id,
                    'numero' => $num ?? (int) $p->id,
                    'label' => $num ? ('P' . $num) : ('P' . (int) $p->id),
                ];
            });

        $rows = collect();
        $indicadoresSeleccionados = collect();
        $indicadoresFortalezas = collect();
        $indicadoresDebilidades = collect();
        $indicadoresRecomendaciones = collect();
        $seleccionAsignatura = null;
        $suggestedCodigos = [
            'Fortaleza' => 100,
            'Debilidad' => 200,
            'Recomendación' => 300,
        ];
        if ($profesor !== '') {
            $assignTable = $this->resolveSalonAsignatProfesorTable($annio);

            $rows = DB::table("{$assignTable} as sap")
                ->join('sweb_salones as s', 's.id', '=', 'sap.salon_id')
                ->join('sweb_asignaturas as a', 'a.id', '=', 'sap.asignatura_id')
                ->leftJoin('sweb_grado_asignat as ga', function ($join) {
                    $join->on('ga.grado_id', '=', 's.grado_id')
                        ->on('ga.asignatura_id', '=', 'sap.asignatura_id');
                })
                ->where('sap.user_id', (int) $profesor)
                ->select(
                    's.id as salon_id',
                    's.grado_id as grado_id',
                    'a.id as asignatura_id',
                    's.nombre as curso_nombre',
                    'a.nombre as asignatura_nombre',
                    'ga.intensidad'
                )
                ->orderBy('s.nombre')
                ->orderBy('a.nombre')
                ->get();

            if ($rows->isNotEmpty()) {
                $salonIds = $rows->pluck('salon_id')->filter()->unique()->values();
                $gradoIds = $rows->pluck('grado_id')->filter()->unique()->values();
                $asignaturaIds = $rows->pluck('asignatura_id')->filter()->unique()->values();

                $estudiantesPorSalon = DB::table('sweb_estudiantes')
                    ->select('salon_id', DB::raw('COUNT(*) as total'))
                    ->whereIn('salon_id', $salonIds)
                    ->where('is_active', 1)
                    ->groupBy('salon_id')
                    ->pluck('total', 'salon_id');

                $indicadoresRaw = DB::table('sweb_indicadores')
                    ->select('grado_id', 'asignatura_id', DB::raw('COUNT(*) as total'))
                    ->where('annio', $annio)
                    ->whereIn('grado_id', $gradoIds)
                    ->whereIn('asignatura_id', $asignaturaIds)
                    ->where('is_active', 1)
                    ->groupBy('grado_id', 'asignatura_id')
                    ->get();

                $indicadoresMap = [];
                foreach ($indicadoresRaw as $r) {
                    $indicadoresMap[((int) $r->grado_id) . '|' . ((int) $r->asignatura_id)] = (int) $r->total;
                }

                $notasRaw = DB::table('sweb_notas_v2')
                    ->select('salon_id', 'asignatura_id', DB::raw('COUNT(*) as total'))
                    ->where('annio', $annio)
                    ->where('profesor_id', (int) $profesor)
                    ->whereIn('salon_id', $salonIds)
                    ->whereIn('asignatura_id', $asignaturaIds)
                    ->groupBy('salon_id', 'asignatura_id')
                    ->get();

                $notasMap = [];
                foreach ($notasRaw as $r) {
                    $notasMap[((int) $r->salon_id) . '|' . ((int) $r->asignatura_id)] = (int) $r->total;
                }

                $rows = $rows->map(function ($row) use ($estudiantesPorSalon, $indicadoresMap, $notasMap) {
                    $salonId = (int) ($row->salon_id ?? 0);
                    $gradoId = (int) ($row->grado_id ?? 0);
                    $asignaturaId = (int) ($row->asignatura_id ?? 0);
                    $row->no_estudiantes = (int) ($estudiantesPorSalon[$salonId] ?? 0);
                    $row->indicadores = (int) ($indicadoresMap[$gradoId . '|' . $asignaturaId] ?? 0);
                    $row->notas = (int) ($notasMap[$salonId . '|' . $asignaturaId] ?? 0);
                    return $row;
                });

                // No autoseleccionar curso/asignatura/periodo.
                // Los indicadores solo deben cargarse por seleccion explicita del usuario.
            }

            if ($cursoId > 0 && $asignaturaId > 0 && $periodoId > 0) {
                $seleccionAsignatura = $rows->first(function ($r) use ($cursoId, $asignaturaId) {
                    return (int) ($r->salon_id ?? 0) === $cursoId
                        && (int) ($r->asignatura_id ?? 0) === $asignaturaId;
                });

                if ($seleccionAsignatura) {
                    $indicadoresSeleccionados = DB::table('sweb_indicadores')
                        ->select('id', 'codigo', 'concepto', 'valorativo')
                        ->where('annio', $annio)
                        ->where('periodo_id', $periodoId)
                        ->where('grado_id', (int) ($seleccionAsignatura->grado_id ?? 0))
                        ->where('asignatura_id', (int) ($seleccionAsignatura->asignatura_id ?? 0))
                        ->where('is_active', 1)
                        ->orderByRaw('CASE WHEN codigo IS NULL OR codigo = "" THEN 9999 ELSE CAST(codigo AS UNSIGNED) END ASC')
                        ->orderBy('id')
                        ->get();

                    $indicadoresFortalezas = $indicadoresSeleccionados->filter(function ($i) {
                        return str_contains(Str::lower((string) ($i->valorativo ?? '')), 'fort');
                    })->values();
                    $indicadoresDebilidades = $indicadoresSeleccionados->filter(function ($i) {
                        return str_contains(Str::lower((string) ($i->valorativo ?? '')), 'debil');
                    })->values();
                    $indicadoresRecomendaciones = $indicadoresSeleccionados->filter(function ($i) {
                        $val = Str::lower((string) ($i->valorativo ?? ''));
                        return str_contains($val, 'recom');
                    })->values();

                    if ($periodoId > 0) {
                        $gradoSel = (int) ($seleccionAsignatura->grado_id ?? 0);
                        $asignaturaSel = (int) ($seleccionAsignatura->asignatura_id ?? 0);
                        $suggestedCodigos = [
                            'Fortaleza' => $this->getNextIndicadorCodigo($annio, $periodoId, $gradoSel, $asignaturaSel, 'Fortaleza'),
                            'Debilidad' => $this->getNextIndicadorCodigo($annio, $periodoId, $gradoSel, $asignaturaSel, 'Debilidad'),
                            'Recomendación' => $this->getNextIndicadorCodigo($annio, $periodoId, $gradoSel, $asignaturaSel, 'Recomendación'),
                        ];
                    }
                }
            }
        }

        return view('admin.academia.asignacion_estado_actual', [
            'title' => 'Asignacion Academica - Estado Actual',
            'annio' => $annio,
            'profesor' => $profesor,
            'profesores' => $profesores,
            'rows' => $rows,
            'isDocenteView' => $isDocenteView,
            'docenteNombre' => $docenteNombre,
            'cursoId' => $cursoId,
            'asignaturaId' => $asignaturaId,
            'periodoId' => $periodoId,
            'periodosIndicadores' => $periodosIndicadores,
            'indicadoresSeleccionados' => $indicadoresSeleccionados,
            'indicadoresFortalezas' => $indicadoresFortalezas,
            'indicadoresDebilidades' => $indicadoresDebilidades,
            'indicadoresRecomendaciones' => $indicadoresRecomendaciones,
            'seleccionAsignatura' => $seleccionAsignatura,
            'suggestedCodigos' => $suggestedCodigos,
        ]);
    }

    public function storeAsignacionIndicador(Request $request)
    {
        $data = $request->validate([
            'annio' => ['required', 'integer', 'min:2000', 'max:2100'],
            'profesor' => ['nullable', 'integer'],
            'curso_id' => ['required', 'integer', Rule::exists('sweb_salones', 'id')],
            'asignatura_id' => ['required', 'integer', Rule::exists('sweb_asignaturas', 'id')],
            'periodo_id' => ['required', 'integer', Rule::exists('sweb_periodos', 'id')],
            'valorativo' => ['required', 'string', 'max:40'],
            'concepto' => ['required', 'string', 'max:4000'],
            'return_url' => ['nullable', 'string', 'max:2000'],
        ]);

        $annio = (int) $data['annio'];
        $cursoId = (int) $data['curso_id'];
        $asignaturaId = (int) $data['asignatura_id'];
        $periodoId = (int) $data['periodo_id'];
        $valorativo = trim((string) $data['valorativo']);

        $salon = DB::table('sweb_salones')
            ->select('id', 'grado_id')
            ->where('id', $cursoId)
            ->first();

        if (! $salon) {
            return back()->with('status', 'No se encontro el curso.');
        }

        $codigo = (string) $this->getNextIndicadorCodigo(
            $annio,
            $periodoId,
            (int) ($salon->grado_id ?? 0),
            $asignaturaId,
            $valorativo
        );

        DB::table('sweb_indicadores')->insert([
            'uuid' => (string) Str::uuid(),
            'annio' => $annio,
            'periodo_id' => $periodoId,
            'grado_id' => (int) ($salon->grado_id ?? 0),
            'asignatura_id' => $asignaturaId,
            'codigo' => $codigo,
            'concepto' => trim((string) $data['concepto']),
            'valorativo' => $valorativo,
            'is_active' => 1,
            'created_by' => (int) session('admin_user_id', 1),
            'updated_by' => (int) session('admin_user_id', 1),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        $returnUrl = trim((string) ($data['return_url'] ?? ''));
        if ($returnUrl !== '') {
            return redirect($returnUrl)->with('status', 'Indicador agregado correctamente.');
        }

        return redirect()->route('admin.academia.asignacion.item', [
            'item' => 'estado-actual',
            'annio' => $annio,
            'profesor' => (int) ($data['profesor'] ?? 0),
            'curso_id' => $cursoId,
            'asignatura_id' => $asignaturaId,
            'periodo_id' => $periodoId,
        ])->with('status', 'Indicador agregado correctamente.');
    }

    public function updateAsignacionIndicador(Request $request, int $id)
    {
        $data = $request->validate([
            'concepto' => ['required', 'string', 'max:4000'],
            'return_url' => ['nullable', 'string', 'max:2000'],
        ]);

        $indicador = DB::table('sweb_indicadores')
            ->select('id')
            ->where('id', $id)
            ->first();

        if (! $indicador) {
            return back()->with('status', 'No se encontro el indicador.');
        }

        DB::table('sweb_indicadores')
            ->where('id', $id)
            ->update([
                'concepto' => trim((string) $data['concepto']),
                'updated_by' => (int) session('admin_user_id', 1),
                'updated_at' => now(),
            ]);

        $returnUrl = trim((string) ($data['return_url'] ?? ''));
        if ($returnUrl !== '') {
            return redirect($returnUrl)->with('status', 'Indicador actualizado correctamente.');
        }

        return back()->with('status', 'Indicador actualizado correctamente.');
    }

    public function saveAsignacionPorCurso(Request $request)
    {
        $annio = (int) $request->input('annio', date('Y'));
        $data = $request->validate([
            'curso' => ['required', 'integer', Rule::exists('sweb_salones', 'id')],
            'profesor_por_asignatura' => ['nullable', 'array'],
        ]);

        $curso = (int) $data['curso'];
        $salon = DB::table('sweb_salones')->select('id', 'grado_id')->where('id', $curso)->first();
        if (! $salon) {
            return redirect()
                ->route('admin.academia.asignacion.item', ['item' => 'por-curso', 'annio' => $annio])
                ->with('status', 'No se encontro el curso.');
        }

        $allowedSubjectIds = DB::table('sweb_grado_asignat')
            ->where('grado_id', (int) $salon->grado_id)
            ->pluck('asignatura_id')
            ->map(fn ($v) => (int) $v)
            ->unique()
            ->values()
            ->all();
        $allowedSet = array_flip($allowedSubjectIds);

        $postedMap = collect($request->input('profesor_por_asignatura', []))
            ->mapWithKeys(function ($profesorId, $asignaturaId) {
                return [(int) $asignaturaId => (int) $profesorId];
            })
            ->filter(fn ($profesorId, $asignaturaId) => $asignaturaId > 0)
            ->all();

        $assignTable = $this->resolveSalonAsignatProfesorTable($annio);

        DB::transaction(function () use ($assignTable, $curso, $allowedSubjectIds, $allowedSet, $postedMap) {
            foreach ($allowedSubjectIds as $subjectId) {
                DB::table($assignTable)
                    ->where('salon_id', $curso)
                    ->where('asignatura_id', $subjectId)
                    ->delete();

                $profesorId = (int) ($postedMap[$subjectId] ?? 0);
                if ($profesorId <= 0) {
                    continue;
                }
                if (!isset($allowedSet[$subjectId])) {
                    continue;
                }

                $validTeacher = DB::table('dm_user')
                    ->where('id', $profesorId)
                    ->whereRaw("TRIM(REPLACE(COALESCE(roll,''), '\n', '')) IN ('docentes','coordinadores')")
                    ->where('is_active', 1)
                    ->exists();
                if (! $validTeacher) {
                    continue;
                }

                DB::table($assignTable)->insert([
                    'salon_id' => $curso,
                    'asignatura_id' => $subjectId,
                    'user_id' => $profesorId,
                    'pend_cal_p1' => 1,
                    'pend_cal_p2' => 1,
                    'pend_cal_p3' => 1,
                    'pend_cal_p4' => 1,
                    'pend_cal_p5' => 1,
                ]);
            }
        });

        return redirect()
            ->route('admin.academia.asignacion.item', [
                'item' => 'por-curso',
                'annio' => $annio,
                'curso' => $curso,
            ])
            ->with('status', 'Asignacion por curso guardada.');
    }

    public function saveAsignacionPorProfesorAsignatura(Request $request)
    {
        $annio = (int) $request->input('annio', date('Y'));
        $data = $request->validate([
            'profesor' => ['required', 'integer'],
            'asignatura' => ['required', 'integer'],
            'salones' => ['nullable', 'array'],
            'salones.*' => ['integer'],
        ]);

        $profesor = (int) $data['profesor'];
        $asignatura = (int) $data['asignatura'];
        $selected = collect($data['salones'] ?? [])->map(fn ($v) => (int) $v)->unique()->values();

        $gradoIds = DB::table('sweb_grado_asignat')
            ->where('asignatura_id', $asignatura)
            ->pluck('grado_id')
            ->unique()
            ->values();

        $allowedSalonIds = DB::table('sweb_salones')
            ->whereIn('grado_id', $gradoIds)
            ->pluck('id')
            ->map(fn ($v) => (int) $v)
            ->all();
        $allowedSet = array_flip($allowedSalonIds);
        $selected = $selected->filter(fn ($id) => isset($allowedSet[$id]))->values();

        $assignTable = $this->resolveSalonAsignatProfesorTable($annio);

        DB::transaction(function () use ($assignTable, $profesor, $asignatura, $allowedSalonIds, $selected) {
            DB::table($assignTable)
                ->where('user_id', $profesor)
                ->where('asignatura_id', $asignatura)
                ->whereIn('salon_id', $allowedSalonIds)
                ->delete();

            if ($selected->isNotEmpty()) {
                $rows = $selected->map(function ($salonId) use ($asignatura, $profesor) {
                    return [
                        'salon_id' => (int) $salonId,
                        'asignatura_id' => $asignatura,
                        'user_id' => $profesor,
                        'pend_cal_p1' => 1,
                        'pend_cal_p2' => 1,
                        'pend_cal_p3' => 1,
                        'pend_cal_p4' => 1,
                        'pend_cal_p5' => 1,
                    ];
                })->all();

                DB::table($assignTable)->insert($rows);
            }
        });

        return redirect()
            ->route('admin.academia.asignacion.item', [
                'item' => 'por-profesor-asignatura',
                'annio' => $annio,
                'profesor' => $profesor,
                'asignatura' => $asignatura,
            ])
            ->with('status', 'Asignacion academica guardada.');
    }

    public function savePeriodosAcademicos(Request $request)
    {
        $annio = (int) $request->input('annio', date('Y'));
        $enabled = collect($request->input('periodos', []))
            ->map(fn ($v) => (int) $v)
            ->filter(fn ($v) => $v >= 1 && $v <= 20)
            ->unique()
            ->sort()
            ->values()
            ->all();

        Cache::forever("academia_periodos_habilitados_{$annio}", $enabled);

        return redirect()
            ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
            ->with('status', 'Periodos academicos actualizados.');
    }

    public function addPeriodoAcademico(Request $request)
    {
        $annio = (int) $request->input('annio', date('Y'));

        $periodNumbers = DB::table('sweb_periodos')
            ->select('periodo')
            ->get()
            ->map(function ($row) {
                if (preg_match('/(\d+)/', (string) $row->periodo, $m)) {
                    return (int) $m[1];
                }

                return null;
            })
            ->filter()
            ->unique()
            ->sort()
            ->values();

        $next = (int) ($periodNumbers->max() ?? 0) + 1;
        if ($next > 20) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
                ->with('status', 'No se pueden crear mas periodos.');
        }

        $lastPeriodDates = DB::table('sweb_periodos')
            ->select('fecha_inicio', 'fecha_fin')
            ->orderByDesc('fecha_fin')
            ->first();

        if ($lastPeriodDates && ! empty($lastPeriodDates->fecha_fin)) {
            $fechaInicio = Carbon::parse($lastPeriodDates->fecha_fin)->addDay();
            $fechaFin = (clone $fechaInicio)->addDays(59);
        } else {
            $fechaInicio = Carbon::create($annio, 1, 1);
            $fechaFin = Carbon::create($annio, 3, 31);
        }

        DB::table('sweb_periodos')->insert([
            'uuid' => (string) Str::uuid(),
            'periodo' => 'Periodo ' . str_pad((string) $next, 2, '0', STR_PAD_LEFT),
            'fecha_inicio' => $fechaInicio->toDateString(),
            'fecha_fin' => $fechaFin->toDateString(),
            'label' => null,
            'rowid' => null,
            'created_by' => 1,
            'updated_by' => 1,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        $key = "academia_periodos_habilitados_{$annio}";
        $enabled = collect(Cache::get($key, []))
            ->map(fn ($v) => (int) $v)
            ->filter(fn ($v) => $v >= 1 && $v <= 20)
            ->push($next)
            ->unique()
            ->sort()
            ->values()
            ->all();
        Cache::forever($key, $enabled);

        return redirect()
            ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
            ->with('status', 'Periodo agregado correctamente.');
    }

    public function editPeriodoAcademico(Request $request, int $numero)
    {
        $annio = (int) $request->input('annio', date('Y'));
        $texto = trim((string) $request->input('texto_periodo', ''));

        if ($numero < 1 || $numero > 20) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
                ->with('status', 'Periodo invalido.');
        }

        if ($texto === '' || mb_strlen($texto) > 20) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
                ->with('status', 'El texto del periodo es obligatorio y maximo 20 caracteres.');
        }

        $row = DB::table('sweb_periodos')
            ->select('id', 'periodo')
            ->get()
            ->first(fn ($r) => $this->extractPeriodNumber((string) $r->periodo) === $numero);

        if (! $row) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
                ->with('status', 'No se encontro el periodo para editar.');
        }

        DB::table('sweb_periodos')
            ->where('id', (int) $row->id)
            ->update([
                'label' => $texto,
                'updated_at' => now(),
                'updated_by' => 1,
            ]);

        return redirect()
            ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
            ->with('status', 'Texto del periodo actualizado.');
    }

    public function deletePeriodoAcademico(Request $request, int $numero)
    {
        $annio = (int) $request->input('annio', date('Y'));
        if ($numero < 1 || $numero > 20) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
                ->with('status', 'Periodo invalido.');
        }

        $row = DB::table('sweb_periodos')
            ->select('id', 'periodo')
            ->get()
            ->first(fn ($r) => $this->extractPeriodNumber((string) $r->periodo) === $numero);

        if (! $row) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
                ->with('status', 'No se encontro el periodo para eliminar.');
        }

        $inUse = DB::table('sweb_notas_v2')->where('periodo_id', (int) $row->id)->exists();
        if ($inUse) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
                ->with('status', 'No se puede eliminar: el periodo ya tiene notas registradas.');
        }

        DB::table('sweb_periodos')->where('id', (int) $row->id)->delete();

        $key = "academia_periodos_habilitados_{$annio}";
        $enabled = collect(Cache::get($key, []))
            ->map(fn ($v) => (int) $v)
            ->reject(fn ($v) => $v === $numero)
            ->values()
            ->all();
        Cache::forever($key, $enabled);

        return redirect()
            ->route('admin.academia.item', ['item' => 'periodos-academicos', 'annio' => $annio])
            ->with('status', 'Periodo eliminado correctamente.');
    }

    private function renderPeriodosAcademicos()
    {
        $annio = (int) request('annio', date('Y'));

        $dbRows = DB::table('sweb_periodos')
            ->select('id', 'periodo', 'label')
            ->orderBy('id')
            ->get();

        $dbMap = $dbRows
            ->mapWithKeys(function ($row) {
                $num = $this->extractPeriodNumber((string) $row->periodo);
                if ($num === null) {
                    return [];
                }

                return [$num => (int) $row->id];
            });
        $dbLabelMap = $dbRows
            ->mapWithKeys(function ($row) {
                $num = $this->extractPeriodNumber((string) $row->periodo);
                if ($num === null) {
                    return [];
                }

                return [$num => trim((string) ($row->label ?? ''))];
            });

        $maxPeriod = max(5, (int) ($dbMap->keys()->max() ?? 5));

        $all = collect(range(1, $maxPeriod))->map(function (int $n) use ($annio, $dbMap, $dbLabelMap) {
            $customLabel = (string) ($dbLabelMap->get($n, ''));
            if ($customLabel !== '') {
                // Si la etiqueta guardada trae año fijo, lo reemplaza por el año seleccionado.
                $customLabel = (string) preg_replace('/\b(19|20)\d{2}\b/u', (string) $annio, $customLabel);
            }
            return [
                'id' => $dbMap->get($n),
                'numero' => $n,
                'label' => $n . ' - ' . $annio,
                'display_label' => $customLabel !== '' ? $customLabel : ($n . ' - ' . $annio),
                'exists' => $dbMap->has($n),
            ];
        });

        $cached = Cache::get("academia_periodos_habilitados_{$annio}");
        if (is_array($cached)) {
            $enabled = collect($cached)->map(fn ($v) => (int) $v)->all();
        } else {
            $enabled = $all->where('exists', true)->pluck('numero')->all();
        }

        return view('admin.academia.periodos', [
            'title' => 'Periodos Academicos',
            'annio' => $annio,
            'periodos' => $all,
            'enabled' => $enabled,
        ]);
    }

    private function renderAreasAcademicas()
    {
        $areas = DB::table('sweb_areas')
            ->select('id', 'nombre', 'orden')
            ->where('is_active', 1)
            ->orderByRaw('CASE WHEN orden = 0 THEN 9999 ELSE orden END ASC')
            ->orderBy('nombre')
            ->get();

        return view('admin.academia.areas', [
            'title' => 'Areas Academicas',
            'areas' => $areas,
        ]);
    }

    private function renderAsignaturas()
    {
        $grado = (string) request('grado', '');

        $rows = DB::table('sweb_asignaturas as a')
            ->leftJoin('sweb_areas as ar', 'ar.id', '=', 'a.area_id')
            ->when($grado !== '', function ($q) use ($grado) {
                $q->join('sweb_grado_asignat as ga', 'ga.asignatura_id', '=', 'a.id')
                    ->where('ga.grado_id', (int) $grado);
            })
            ->where('a.is_active', 1)
            ->select('a.id', 'a.nombre', 'a.orden', 'ar.nombre as area_nombre')
            ->distinct()
            ->orderBy('a.nombre')
            ->paginate(25)
            ->withQueryString();

        $grados = DB::table('sweb_grados')
            ->select('id', 'nombre', 'orden')
            ->where('is_active', 1)
            ->orderByRaw('CASE WHEN orden IS NULL THEN 9999 ELSE orden END ASC')
            ->orderBy('nombre')
            ->get();

        return view('admin.academia.asignaturas', [
            'title' => 'Asignaturas',
            'rows' => $rows,
            'grados' => $grados,
            'grado' => $grado,
        ]);
    }

    public function addAreaAcademica(Request $request)
    {
        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:80', Rule::unique('sweb_areas', 'nombre')],
            'orden' => ['nullable', 'integer', 'min:0', 'max:999'],
        ]);

        $nombre = trim((string) $data['nombre']);
        $orden = (int) ($data['orden'] ?? 0);
        DB::table('sweb_areas')->insert([
            'uuid' => (string) Str::uuid(),
            'nombre' => $nombre,
            'orden' => $orden,
            'created_by' => 1,
            'updated_by' => 1,
            'created_at' => now(),
            'updated_at' => now(),
            'is_active' => 1,
        ]);

        return redirect()
            ->route('admin.academia.item', ['item' => 'areas-academicas'])
            ->with('status', 'Area academica agregada correctamente.');
    }

    public function createAreaAcademica()
    {
        return view('admin.academia.areas_create', [
            'title' => 'Detalle de Area Academica',
        ]);
    }

    public function showAreaAcademica(int $id)
    {
        $area = DB::table('sweb_areas')
            ->select('id', 'nombre', 'orden')
            ->where('id', $id)
            ->first();

        abort_unless($area, 404);

        $asignaturas = DB::table('sweb_asignaturas')
            ->select('id', 'nombre', 'abrev', 'is_active')
            ->where('area_id', $id)
            ->orderBy('nombre')
            ->get();

        $availableAsignaturas = DB::table('sweb_asignaturas')
            ->select('id', 'nombre', 'abrev', 'area_id')
            ->where(function ($q) use ($id) {
                $q->whereNull('area_id')
                    ->orWhere('area_id', '<>', $id);
            })
            ->orderBy('nombre')
            ->limit(300)
            ->get();

        return view('admin.academia.areas_show', [
            'title' => 'Detalle de Area Academica',
            'area' => $area,
            'asignaturas' => $asignaturas,
            'availableAsignaturas' => $availableAsignaturas,
        ]);
    }

    public function updateAreaAcademica(Request $request, int $id)
    {
        $area = DB::table('sweb_areas')->where('id', $id)->first();
        if (! $area) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'areas-academicas'])
                ->with('status', 'No se encontro el area.');
        }

        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:80', Rule::unique('sweb_areas', 'nombre')->ignore($id)],
            'orden' => ['required', 'integer', 'min:0', 'max:999'],
        ]);

        DB::table('sweb_areas')
            ->where('id', $id)
            ->update([
                'nombre' => trim((string) $data['nombre']),
                'orden' => (int) $data['orden'],
                'updated_by' => 1,
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.academia.areas.show', ['id' => $id])
            ->with('status', 'Area academica actualizada.');
    }

    public function attachAsignaturaArea(Request $request, int $id)
    {
        $area = DB::table('sweb_areas')->where('id', $id)->first();
        if (! $area) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'areas-academicas'])
                ->with('status', 'No se encontro el area.');
        }

        $data = $request->validate([
            'asignatura_id' => ['required', 'integer'],
        ]);

        $asignatura = DB::table('sweb_asignaturas')
            ->select('id')
            ->where('id', (int) $data['asignatura_id'])
            ->first();

        if (! $asignatura) {
            return redirect()
                ->route('admin.academia.areas.show', ['id' => $id])
                ->with('status', 'No se encontro la asignatura seleccionada.');
        }

        DB::table('sweb_asignaturas')
            ->where('id', (int) $asignatura->id)
            ->update([
                'area_id' => $id,
                'updated_by' => 1,
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.academia.areas.show', ['id' => $id])
            ->with('status', 'Asignatura agregada al area.');
    }

    public function detachAsignaturaArea(int $id, int $asignaturaId)
    {
        $area = DB::table('sweb_areas')->where('id', $id)->first();
        if (! $area) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'areas-academicas'])
                ->with('status', 'No se encontro el area.');
        }

        $asignatura = DB::table('sweb_asignaturas')
            ->select('id', 'area_id')
            ->where('id', $asignaturaId)
            ->first();

        if (! $asignatura || (int) $asignatura->area_id !== $id) {
            return redirect()
                ->route('admin.academia.areas.show', ['id' => $id])
                ->with('status', 'La asignatura no pertenece a esta area.');
        }

        DB::table('sweb_asignaturas')
            ->where('id', $asignaturaId)
            ->update([
                'area_id' => null,
                'updated_by' => 1,
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.academia.areas.show', ['id' => $id])
            ->with('status', 'Asignatura quitada del area.');
    }

    public function editAreaAcademica(Request $request, int $id)
    {
        $area = DB::table('sweb_areas')->where('id', $id)->first();
        if (! $area) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'areas-academicas'])
                ->with('status', 'No se encontro el area para editar.');
        }

        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:80', Rule::unique('sweb_areas', 'nombre')->ignore($id)],
            'orden' => ['nullable', 'integer', 'min:0', 'max:999'],
        ]);

        DB::table('sweb_areas')
            ->where('id', $id)
            ->update([
                'nombre' => trim((string) $data['nombre']),
                'orden' => (int) ($data['orden'] ?? 0),
                'updated_by' => 1,
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.academia.item', ['item' => 'areas-academicas'])
            ->with('status', 'Area academica actualizada.');
    }

    public function deleteAreaAcademica(int $id)
    {
        $area = DB::table('sweb_areas')->where('id', $id)->first();
        if (! $area) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'areas-academicas'])
                ->with('status', 'No se encontro el area para eliminar.');
        }

        $inUse = DB::table('sweb_asignaturas')
            ->where('area_id', $id)
            ->exists();

        if ($inUse) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'areas-academicas'])
                ->with('status', 'No se puede eliminar: el area tiene asignaturas activas asociadas.');
        }

        DB::table('sweb_areas')->where('id', $id)->delete();

        return redirect()
            ->route('admin.academia.item', ['item' => 'areas-academicas'])
            ->with('status', 'Area academica eliminada.');
    }

    public function createAsignatura()
    {
        $areas = DB::table('sweb_areas')
            ->select('id', 'nombre')
            ->where('is_active', 1)
            ->orderBy('nombre')
            ->get();

        return view('admin.academia.asignaturas_show', [
            'title' => 'Detalle de Asignatura',
            'isCreate' => true,
            'asignatura' => (object) [
                'id' => null,
                'nombre' => '',
                'abrev' => '',
                'orden' => 0,
                'area_id' => null,
                'is_active' => 1,
                'calc_prom' => 1,
            ],
            'areas' => $areas,
            'grados' => collect(),
        ]);
    }

    public function storeAsignatura(Request $request)
    {
        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:50', Rule::unique('sweb_asignaturas', 'nombre')],
            'abrev' => ['required', 'string', 'max:10', Rule::unique('sweb_asignaturas', 'abrev')],
            'orden' => ['nullable', 'integer', 'min:0', 'max:9999'],
            'area_id' => ['nullable', 'integer', Rule::exists('sweb_areas', 'id')],
            'is_active' => ['nullable', 'boolean'],
            'calc_prom' => ['nullable', 'boolean'],
        ]);

        DB::table('sweb_asignaturas')->insert([
            'uuid' => (string) Str::uuid(),
            'nombre' => trim((string) $data['nombre']),
            'abrev' => mb_strtoupper(trim((string) $data['abrev'])),
            'orden' => (int) ($data['orden'] ?? 0),
            'area_id' => $data['area_id'] !== null ? (int) $data['area_id'] : null,
            'is_active' => (int) ($data['is_active'] ?? 0),
            'calc_prom' => (int) ($data['calc_prom'] ?? 0),
            'created_by' => 1,
            'updated_by' => 1,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        $id = (int) DB::getPdo()->lastInsertId();

        return redirect()
            ->route('admin.academia.asignaturas.show', ['id' => $id])
            ->with('status', 'Asignatura creada correctamente.');
    }

    public function showAsignatura(int $id)
    {
        $asignatura = DB::table('sweb_asignaturas')
            ->select('id', 'nombre', 'abrev', 'orden', 'area_id', 'is_active', 'calc_prom')
            ->where('id', $id)
            ->first();

        abort_unless($asignatura, 404);

        $areas = DB::table('sweb_areas')
            ->select('id', 'nombre')
            ->where('is_active', 1)
            ->orderBy('nombre')
            ->get();

        $grados = DB::table('sweb_grado_asignat as ga')
            ->join('sweb_grados as g', 'g.id', '=', 'ga.grado_id')
            ->where('ga.asignatura_id', $id)
            ->select('g.nombre')
            ->orderBy('g.orden')
            ->orderBy('g.nombre')
            ->get();

        return view('admin.academia.asignaturas_show', [
            'title' => 'Detalle de Asignatura',
            'isCreate' => false,
            'asignatura' => $asignatura,
            'areas' => $areas,
            'grados' => $grados,
        ]);
    }

    public function updateAsignatura(Request $request, int $id)
    {
        $asignatura = DB::table('sweb_asignaturas')->where('id', $id)->first();
        if (! $asignatura) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'asignaturas'])
                ->with('status', 'No se encontro la asignatura.');
        }

        $data = $request->validate([
            'nombre' => ['required', 'string', 'max:50', Rule::unique('sweb_asignaturas', 'nombre')->ignore($id)],
            'abrev' => ['required', 'string', 'max:10', Rule::unique('sweb_asignaturas', 'abrev')->ignore($id)],
            'orden' => ['nullable', 'integer', 'min:0', 'max:9999'],
            'area_id' => ['nullable', 'integer', Rule::exists('sweb_areas', 'id')],
            'is_active' => ['nullable', 'boolean'],
            'calc_prom' => ['nullable', 'boolean'],
        ]);

        DB::table('sweb_asignaturas')
            ->where('id', $id)
            ->update([
                'nombre' => trim((string) $data['nombre']),
                'abrev' => mb_strtoupper(trim((string) $data['abrev'])),
                'orden' => (int) ($data['orden'] ?? 0),
                'area_id' => $data['area_id'] !== null ? (int) $data['area_id'] : null,
                'is_active' => (int) ($data['is_active'] ?? 0),
                'calc_prom' => (int) ($data['calc_prom'] ?? 0),
                'updated_by' => 1,
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.academia.asignaturas.show', ['id' => $id])
            ->with('status', 'Asignatura actualizada.');
    }

    public function deleteAsignatura(int $id)
    {
        $asignatura = DB::table('sweb_asignaturas')->where('id', $id)->first();
        if (! $asignatura) {
            return redirect()
                ->route('admin.academia.item', ['item' => 'asignaturas'])
                ->with('status', 'No se encontro la asignatura.');
        }

        $inUseNotas = DB::table('sweb_notas_v2')->where('asignatura_id', $id)->exists();
        $inUseGrados = DB::table('sweb_grado_asignat')->where('asignatura_id', $id)->exists();
        if ($inUseNotas || $inUseGrados) {
            return redirect()
                ->route('admin.academia.asignaturas.show', ['id' => $id])
                ->with('status', 'No se puede eliminar: la asignatura tiene informacion relacionada.');
        }

        DB::table('sweb_asignaturas')->where('id', $id)->delete();

        return redirect()
            ->route('admin.academia.item', ['item' => 'asignaturas'])
            ->with('status', 'Asignatura eliminada.');
    }

    public function showObservacionesGenerales(Request $request)
    {
        $q = trim((string) $request->query('q', ''));
        $role = Str::lower((string) session('admin_role', ''));
        $sessionUserId = (int) session('admin_user_id', 0);
        $isDocente = $role === 'docentes';

        $annioParam = $request->query('annio');
        if ($annioParam !== null && $annioParam !== '') {
            $annio = (int) $annioParam;
        } elseif ($isDocente && $sessionUserId > 0) {
            $annio = (int) (DB::table('sweb_estudiantes_reg_observ_gen')
                ->where('created_by', $sessionUserId)
                ->max('annio') ?: date('Y'));
        } elseif (in_array($role, ['coordinadores', 'admin', 'superadmin'], true)) {
            $annio = (int) date('Y') - 1;
        } else {
            $annio = (int) date('Y');
        }

        [$estudiantesTable] = $this->resolveStudentTablesByYear($annio);

        $docenteAllowedSalonIds = [];
        if ($isDocente && $sessionUserId > 0) {
            $docenteAllowedSalonIds = $this->getDocenteAllowedSalonIdsForRegistro($annio, $sessionUserId);
        }

        $periodos = DB::table('sweb_periodos')
            ->select('id', 'periodo', 'label')
            ->orderBy('id')
            ->get();

        $docenteAllowedSalonIds = [];
        if ($isDocente && $sessionUserId > 0) {
            $docenteAllowedSalonIds = $this->getDocenteAllowedSalonIdsForRegistro($annio, $sessionUserId);
        }

        $docenteAllowedSalonIds = [];
        if ($isDocente && $sessionUserId > 0) {
            $docenteAllowedSalonIds = $this->getDocenteAllowedSalonIdsForRegistro($annio, $sessionUserId);
        }

        $students = DB::table("{$estudiantesTable} as e")
            ->leftJoin('sweb_salones as s', 's.id', '=', 'e.salon_id')
            ->select(
                'e.id',
                'e.contabilidad_id',
                'e.salon_id',
                DB::raw("TRIM(CONCAT(COALESCE(e.apellido1,''),' ',COALESCE(e.apellido2,''),' ',COALESCE(e.nombres,''))) as full_name"),
                DB::raw("COALESCE(s.nombre,'') as salon")
            )
            ->where('e.is_active', 1)
            ->when($isDocente && $sessionUserId > 0, function ($query) use ($docenteAllowedSalonIds) {
                if (empty($docenteAllowedSalonIds)) {
                    $query->whereRaw('1 = 0');
                } else {
                    $query->whereIn('e.salon_id', $docenteAllowedSalonIds);
                }
            })
            ->orderBy('e.apellido1')
            ->orderBy('e.apellido2')
            ->orderBy('e.nombres')
            ->limit(1500)
            ->get();

        $rows = DB::table('sweb_estudiantes_reg_observ_gen as o')
            ->leftJoin("{$estudiantesTable} as e", 'e.id', '=', 'o.estudiante_id')
            ->leftJoin('sweb_salones as s', 's.id', '=', 'o.salon_id')
            ->leftJoin('sweb_periodos as p', 'p.id', '=', 'o.periodo_id')
            ->leftJoin('dm_user as u', 'u.id', '=', 'o.created_by')
            ->where('o.annio', $annio)
            ->when($isDocente && $sessionUserId > 0, function ($query) use ($sessionUserId) {
                $query->where('o.created_by', $sessionUserId);
            })
            ->when($q !== '', function ($query) use ($q) {
                $query->where(function ($sub) use ($q) {
                    $sub->where('o.asunto', 'like', "%{$q}%")
                        ->orWhere('o.tipo_reg', 'like', "%{$q}%")
                        ->orWhereRaw("CONCAT(COALESCE(e.nombres,''),' ',COALESCE(e.apellido1,''),' ',COALESCE(e.apellido2,'')) LIKE ?", ["%{$q}%"]);
                });
            })
            ->select(
                'o.id',
                'o.fecha',
                'o.annio',
                'o.tipo_reg',
                'o.asunto',
                'o.foto_acudiente',
                'o.estudiante_id',
                'o.periodo_id',
                'o.salon_id',
                'p.periodo',
                DB::raw("TRIM(CONCAT(COALESCE(e.nombres,''),' ',COALESCE(e.apellido1,''),' ',COALESCE(e.apellido2,''))) as estudiante"),
                DB::raw("COALESCE(s.nombre,'') as salon"),
                DB::raw("TRIM(CONCAT(COALESCE(u.nombres,''),' ',COALESCE(u.apellido1,''),' ',COALESCE(u.apellido2,''))) as docente")
            )
            ->orderByDesc('o.fecha')
            ->orderByDesc('o.id')
            ->paginate(30)
            ->withQueryString();

        return view('admin.observador.observaciones_generales', [
            'title' => 'Registros de Observaciones Generales',
            'annio' => $annio,
            'q' => $q,
            'rows' => $rows,
            'students' => $students,
            'periodos' => $periodos,
            'isDocente' => $isDocente,
            'canDeleteObservacion' => in_array($role, ['coordinadores', 'admin', 'superadmin'], true),
        ]);
    }

    public function storeObservacionGeneral(Request $request)
    {
        $role = Str::lower((string) session('admin_role', ''));
        $sessionUserId = (int) session('admin_user_id', 0);
        $isDocente = $role === 'docentes';

        $data = $request->validate([
            'annio' => ['required', 'integer', 'min:2000', 'max:2100'],
            'tipo_reg' => ['required', 'string', 'max:10'],
            'estudiante_id' => ['required', 'integer'],
            'periodo_id' => ['required', 'integer', Rule::exists('sweb_periodos', 'id')],
            'fecha' => ['required', 'date'],
            'asunto' => ['required', 'string', 'max:20000'],
            'evidencia' => ['nullable', 'file', 'mimes:jpg,jpeg,png,pdf', 'max:5120'],
        ]);

        [$estudiantesTable] = $this->resolveStudentTablesByYear((int) $data['annio']);
        $estudiante = DB::table($estudiantesTable)
            ->select('id', 'salon_id')
            ->where('id', (int) $data['estudiante_id'])
            ->first();

        if (! $estudiante) {
            return back()->with('status', 'No se encontro el estudiante seleccionado.');
        }

        $salonId = (int) ($estudiante->salon_id ?? 0);
        $gradoId = (int) (DB::table('sweb_salones')->where('id', $salonId)->value('grado_id') ?? 0);

        if ($isDocente && $sessionUserId > 0) {
            $salonIds = $this->getDocenteSalonIdsByAnnio((int) $data['annio'], $sessionUserId);
            if (! in_array($salonId, $salonIds, true)) {
                return back()->with('status', 'Solo puedes registrar observaciones de estudiantes en tu carga académica.');
            }
        }

        $evidenciaPath = $this->storeObservacionFile($request->file('evidencia'));

        DB::table('sweb_estudiantes_reg_observ_gen')->insert([
            'uuid' => (string) Str::uuid(),
            'tipo_reg' => mb_strtoupper(trim((string) $data['tipo_reg'])),
            'estudiante_id' => (int) $data['estudiante_id'],
            'annio' => (int) $data['annio'],
            'periodo_id' => (int) $data['periodo_id'],
            'grado_id' => $gradoId,
            'salon_id' => $salonId,
            'fecha' => $data['fecha'],
            'asunto' => trim((string) $data['asunto']),
            'acudiente' => '',
            'foto_acudiente' => $evidenciaPath,
            'director' => '',
            'foto_director' => '',
            'created_by' => (int) session('admin_user_id', 1),
            'updated_by' => (int) session('admin_user_id', 1),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return redirect()
            ->route('admin.observador.observaciones.index', ['annio' => (int) $data['annio']])
            ->with('status', 'Guardado correctamente');
    }

    public function editObservacionGeneral(int $id)
    {
        $obs = DB::table('sweb_estudiantes_reg_observ_gen')->where('id', $id)->first();
        abort_unless($obs, 404);

        $role = Str::lower((string) session('admin_role', ''));
        $sessionUserId = (int) session('admin_user_id', 0);
        if ($role === 'docentes' && (int) ($obs->created_by ?? 0) !== $sessionUserId) {
            abort(403, 'No tienes permisos para editar esta observacion.');
        }

        $annio = (int) ($obs->annio ?? date('Y'));
        [$estudiantesTable] = $this->resolveStudentTablesByYear($annio);
        $docenteAllowedSalonIds = [];
        if ($isDocente && $sessionUserId > 0) {
            $docenteAllowedSalonIds = $this->getDocenteAllowedSalonIdsForRegistro($annio, $sessionUserId);
        }

        $periodos = DB::table('sweb_periodos')
            ->select('id', 'periodo', 'label')
            ->orderBy('id')
            ->get();

        $students = DB::table("{$estudiantesTable} as e")
            ->leftJoin('sweb_salones as s', 's.id', '=', 'e.salon_id')
            ->select(
                'e.id',
                'e.contabilidad_id',
                DB::raw("TRIM(CONCAT(COALESCE(e.apellido1,''),' ',COALESCE(e.apellido2,''),' ',COALESCE(e.nombres,''))) as full_name"),
                DB::raw("COALESCE(s.nombre,'') as salon")
            )
            ->where('e.is_active', 1)
            ->when($role === 'docentes' && $sessionUserId > 0, function ($query) use ($annio, $sessionUserId, $obs) {
                $salonIds = $this->getDocenteSalonIdsByAnnio($annio, $sessionUserId);
                $obsSalonId = (int) ($obs->salon_id ?? 0);
                if ($obsSalonId > 0 && ! in_array($obsSalonId, $salonIds, true)) {
                    $salonIds[] = $obsSalonId;
                }
                if (empty($salonIds)) {
                    $query->whereRaw('1 = 0');
                } else {
                    $query->whereIn('e.salon_id', $salonIds);
                }
            })
            ->orderBy('e.apellido1')
            ->orderBy('e.apellido2')
            ->orderBy('e.nombres')
            ->limit(1500)
            ->get();

        return view('admin.observador.observaciones_generales_edit', [
            'title' => 'Editar Observación General',
            'obs' => $obs,
            'periodos' => $periodos,
            'students' => $students,
        ]);
    }

    public function updateObservacionGeneral(Request $request, int $id)
    {
        $obs = DB::table('sweb_estudiantes_reg_observ_gen')->where('id', $id)->first();
        if (! $obs) {
            return redirect()->route('admin.observador.observaciones.index')->with('status', 'No se encontro el registro.');
        }

        $role = Str::lower((string) session('admin_role', ''));
        $sessionUserId = (int) session('admin_user_id', 0);
        $isDocente = $role === 'docentes';
        if ($role === 'docentes' && (int) ($obs->created_by ?? 0) !== $sessionUserId) {
            abort(403, 'No tienes permisos para actualizar esta observacion.');
        }

        $data = $request->validate([
            'annio' => ['required', 'integer', 'min:2000', 'max:2100'],
            'tipo_reg' => ['required', 'string', 'max:10'],
            'estudiante_id' => ['required', 'integer'],
            'periodo_id' => ['required', 'integer', Rule::exists('sweb_periodos', 'id')],
            'fecha' => ['required', 'date'],
            'asunto' => ['required', 'string', 'max:20000'],
            'evidencia' => ['nullable', 'file', 'mimes:jpg,jpeg,png,pdf', 'max:5120'],
        ]);

        [$estudiantesTable] = $this->resolveStudentTablesByYear((int) $data['annio']);
        $estudiante = DB::table($estudiantesTable)
            ->select('id', 'salon_id')
            ->where('id', (int) $data['estudiante_id'])
            ->first();
        if (! $estudiante) {
            return back()->with('status', 'No se encontro el estudiante seleccionado.');
        }

        $salonId = (int) ($estudiante->salon_id ?? 0);
        $gradoId = (int) (DB::table('sweb_salones')->where('id', $salonId)->value('grado_id') ?? 0);

        if ($isDocente && $sessionUserId > 0) {
            $salonIds = $this->getDocenteSalonIdsByAnnio((int) $data['annio'], $sessionUserId);
            if (! in_array($salonId, $salonIds, true)) {
                return back()->with('status', 'Solo puedes seleccionar estudiantes de tu carga académica.');
            }
        }

        $evidenciaPath = $obs->foto_acudiente ?: '';
        if ($request->hasFile('evidencia')) {
            $evidenciaPath = $this->storeObservacionFile($request->file('evidencia'));
        }

        DB::table('sweb_estudiantes_reg_observ_gen')
            ->where('id', $id)
            ->update([
                'tipo_reg' => mb_strtoupper(trim((string) $data['tipo_reg'])),
                'estudiante_id' => (int) $data['estudiante_id'],
                'annio' => (int) $data['annio'],
                'periodo_id' => (int) $data['periodo_id'],
                'grado_id' => $gradoId,
                'salon_id' => $salonId,
                'fecha' => $data['fecha'],
                'asunto' => trim((string) $data['asunto']),
                'foto_acudiente' => $evidenciaPath,
                'updated_by' => (int) session('admin_user_id', 1),
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.observador.observaciones.index', ['annio' => (int) $data['annio']])
            ->with('status', 'Guardado correctamente');
    }

    public function deleteObservacionGeneral(int $id)
    {
        $role = Str::lower((string) session('admin_role', ''));
        if (! in_array($role, ['coordinadores', 'admin', 'superadmin'], true)) {
            abort(403, 'No tienes permisos para eliminar observaciones.');
        }

        $obs = DB::table('sweb_estudiantes_reg_observ_gen')
            ->select('id', 'annio')
            ->where('id', $id)
            ->first();

        if (! $obs) {
            return redirect()
                ->route('admin.observador.observaciones.index')
                ->with('status', 'No se encontró la observación para eliminar.');
        }

        DB::table('sweb_estudiantes_reg_observ_gen')->where('id', $id)->delete();

        return redirect()
            ->route('admin.observador.observaciones.index', ['annio' => (int) ($obs->annio ?? date('Y'))])
            ->with('status', 'Observación eliminada correctamente.');
    }

    public function showRegistroAcademico(Request $request)
    {
        $q = trim((string) $request->query('q', ''));
        $role = Str::lower((string) session('admin_role', ''));
        $sessionUserId = (int) session('admin_user_id', 0);
        abort_unless($this->canAccessRegistroAcademico($role, $sessionUserId), 403);
        $isDocente = $role === 'docentes';

        $annioParam = $request->query('annio');
        if ($annioParam !== null && $annioParam !== '') {
            $annio = (int) $annioParam;
        } elseif ($isDocente && $sessionUserId > 0) {
            $annio = (int) (DB::table('sweb_estudiantes_reg_desacacom as r')
                ->leftJoin('sweb_salones as s', 's.id', '=', 'r.salon_id')
                ->where(function ($query) use ($sessionUserId) {
                    $query->where('r.created_by', $sessionUserId)
                        ->orWhere('s.director_id', $sessionUserId);
                })
                ->max('r.annio') ?: 0);
            if ($annio <= 0) {
                $annio = (int) date('Y') - 1;
            }
        } elseif (in_array($role, ['coordinadores', 'admin', 'superadmin'], true)) {
            $annio = (int) date('Y') - 1;
        } else {
            $annio = (int) date('Y');
        }

        [$estudiantesTable] = $this->resolveStudentTablesByYear($annio);
        $docenteAllowedSalonIds = [];
        if ($isDocente && $sessionUserId > 0) {
            $docenteAllowedSalonIds = $this->getDocenteAllowedSalonIdsForRegistro($annio, $sessionUserId);
        }

        $periodos = DB::table('sweb_periodos')
            ->select('id', 'periodo', 'label')
            ->orderBy('id')
            ->get();

        $students = DB::table("{$estudiantesTable} as e")
            ->leftJoin('sweb_salones as s', 's.id', '=', 'e.salon_id')
            ->select(
                'e.id',
                'e.contabilidad_id',
                'e.salon_id',
                DB::raw("TRIM(CONCAT(COALESCE(e.apellido1,''),' ',COALESCE(e.apellido2,''),' ',COALESCE(e.nombres,''))) as full_name"),
                DB::raw("COALESCE(s.nombre,'') as salon")
            )
            ->where('e.is_active', 1)
            ->when($isDocente && $sessionUserId > 0, function ($query) use ($docenteAllowedSalonIds) {
                if (empty($docenteAllowedSalonIds)) {
                    $query->whereRaw('1 = 0');
                } else {
                    $query->whereIn('e.salon_id', $docenteAllowedSalonIds);
                }
            })
            ->orderBy('e.apellido1')
            ->orderBy('e.apellido2')
            ->orderBy('e.nombres')
            ->limit(1500)
            ->get();

        $rows = DB::table('sweb_estudiantes_reg_desacacom as r')
            ->leftJoin("{$estudiantesTable} as e", 'e.id', '=', 'r.estudiante_id')
            ->leftJoin('sweb_salones as s', 's.id', '=', 'r.salon_id')
            ->leftJoin('sweb_periodos as p', 'p.id', '=', 'r.periodo_id')
            ->leftJoin('dm_user as u', 'u.id', '=', 'r.created_by')
            ->where('r.annio', $annio)
            ->when($isDocente && $sessionUserId > 0, function ($query) use ($sessionUserId, $docenteAllowedSalonIds) {
                $query->where(function ($sub) use ($sessionUserId, $docenteAllowedSalonIds) {
                    $sub->where('r.created_by', $sessionUserId);
                    if (! empty($docenteAllowedSalonIds)) {
                        $sub->orWhereIn('r.salon_id', $docenteAllowedSalonIds);
                    }
                });
            })
            ->when($q !== '', function ($query) use ($q) {
                $query->where(function ($sub) use ($q) {
                    $sub->where('r.fortalezas', 'like', "%{$q}%")
                        ->orWhere('r.dificultades', 'like', "%{$q}%")
                        ->orWhere('r.compromisos', 'like', "%{$q}%")
                        ->orWhereRaw("CONCAT(COALESCE(e.nombres,''),' ',COALESCE(e.apellido1,''),' ',COALESCE(e.apellido2,'')) LIKE ?", ["%{$q}%"]);
                });
            })
            ->select(
                'r.id',
                'r.fecha',
                'r.annio',
                'r.created_by',
                'r.fortalezas',
                'r.dificultades',
                'r.compromisos',
                'r.foto_acudiente',
                'r.estudiante_id',
                'r.periodo_id',
                'r.salon_id',
                'r.acudiente',
                'p.periodo',
                DB::raw("TRIM(CONCAT(COALESCE(e.nombres,''),' ',COALESCE(e.apellido1,''),' ',COALESCE(e.apellido2,''))) as estudiante"),
                DB::raw("COALESCE(s.nombre,'') as salon"),
                DB::raw("TRIM(CONCAT(COALESCE(u.nombres,''),' ',COALESCE(u.apellido1,''),' ',COALESCE(u.apellido2,''))) as docente")
            )
            ->orderByDesc('r.fecha')
            ->orderByDesc('r.id')
            ->paginate(30)
            ->withQueryString();

        $directorCursosByUser = [];
        $docenteIds = collect($rows->items())
            ->pluck('created_by')
            ->map(fn ($v) => (int) $v)
            ->filter(fn ($v) => $v > 0)
            ->unique()
            ->values()
            ->all();

        if (! empty($docenteIds)) {
            $directorCursosByUser = DB::table('sweb_salones')
                ->select('director_id', 'nombre')
                ->where('is_active', 1)
                ->whereIn('director_id', $docenteIds)
                ->orderBy('nombre')
                ->get()
                ->groupBy('director_id')
                ->map(function ($rows) {
                    return $rows->pluck('nombre')
                        ->map(fn ($v) => trim((string) $v))
                        ->filter(fn ($v) => $v !== '')
                        ->unique()
                        ->values()
                        ->all();
                })
                ->toArray();
        }

        return view('admin.observador.registro_academico', [
            'title' => 'Registros de Desempeno Academico',
            'annio' => $annio,
            'q' => $q,
            'rows' => $rows,
            'students' => $students,
            'periodos' => $periodos,
            'isDocente' => $isDocente,
            'canDeleteRegistroAcademico' => in_array($role, ['coordinadores', 'admin', 'superadmin'], true),
            'directorCursosByUser' => $directorCursosByUser,
        ]);
    }

    public function storeRegistroAcademico(Request $request)
    {
        $role = Str::lower((string) session('admin_role', ''));
        $sessionUserId = (int) session('admin_user_id', 0);
        abort_unless($this->canAccessRegistroAcademico($role, $sessionUserId), 403);
        $isDocente = $role === 'docentes';

        $data = $request->validate([
            'annio' => ['required', 'integer', 'min:2000', 'max:2100'],
            'estudiante_id' => ['required', 'integer'],
            'periodo_id' => ['required', 'integer', Rule::exists('sweb_periodos', 'id')],
            'fecha' => ['required', 'date'],
            'fortalezas' => ['nullable', 'string', 'max:1024'],
            'dificultades' => ['nullable', 'string', 'max:1024'],
            'compromisos' => ['nullable', 'string', 'max:1024'],
            'evidencia' => ['nullable', 'file', 'mimes:jpg,jpeg,png,pdf', 'max:5120'],
        ]);

        $fortalezas = trim((string) ($data['fortalezas'] ?? ''));
        $dificultades = trim((string) ($data['dificultades'] ?? ''));
        $compromisos = trim((string) ($data['compromisos'] ?? ''));
        if ($fortalezas === '' && $dificultades === '' && $compromisos === '') {
            return back()->with('status', 'Debes diligenciar fortalezas, dificultades o compromisos.');
        }

        [$estudiantesTable] = $this->resolveStudentTablesByYear((int) $data['annio']);
        $estudiante = DB::table($estudiantesTable)
            ->select('id', 'salon_id')
            ->where('id', (int) $data['estudiante_id'])
            ->first();

        if (! $estudiante) {
            return back()->with('status', 'No se encontro el estudiante seleccionado.');
        }

        $salonId = (int) ($estudiante->salon_id ?? 0);
        $gradoId = (int) (DB::table('sweb_salones')->where('id', $salonId)->value('grado_id') ?? 0);

        if ($isDocente && $sessionUserId > 0) {
            $salonIds = $this->getDocenteAllowedSalonIdsForRegistro((int) $data['annio'], $sessionUserId);
            if (! in_array($salonId, $salonIds, true)) {
                return back()->with('status', 'Solo puedes registrar estudiantes de tus cursos asignados o dirigidos.');
            }
        }

        $evidenciaPath = $this->storeObservacionFile($request->file('evidencia'));

        DB::table('sweb_estudiantes_reg_desacacom')->insert([
            'uuid' => (string) Str::uuid(),
            'estudiante_id' => (int) $data['estudiante_id'],
            'annio' => (int) $data['annio'],
            'periodo_id' => (int) $data['periodo_id'],
            'grado_id' => $gradoId,
            'salon_id' => $salonId,
            'fortalezas' => $fortalezas,
            'dificultades' => $dificultades,
            'compromisos' => $compromisos,
            'fecha' => $data['fecha'],
            'acudiente' => '',
            'foto_acudiente' => $evidenciaPath,
            'director' => '',
            'foto_director' => '',
            'created_by' => (int) session('admin_user_id', 1),
            'updated_by' => (int) session('admin_user_id', 1),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return redirect()
            ->route('admin.observador.registro_academico.index', ['annio' => (int) $data['annio']])
            ->with('status', 'Guardado correctamente');
    }

    public function editRegistroAcademico(int $id)
    {
        $reg = DB::table('sweb_estudiantes_reg_desacacom')->where('id', $id)->first();
        abort_unless($reg, 404);

        $role = Str::lower((string) session('admin_role', ''));
        $sessionUserId = (int) session('admin_user_id', 0);
        abort_unless($this->canAccessRegistroAcademico($role, $sessionUserId), 403);
        if ($role === 'docentes' && (int) ($reg->created_by ?? 0) !== $sessionUserId) {
            $isDirectorDelSalon = DB::table('sweb_salones')
                ->where('id', (int) ($reg->salon_id ?? 0))
                ->where('director_id', $sessionUserId)
                ->exists();
            if (! $isDirectorDelSalon) {
                abort(403, 'No tienes permisos para editar este registro.');
            }
        }

        $annio = (int) ($reg->annio ?? date('Y'));
        [$estudiantesTable] = $this->resolveStudentTablesByYear($annio);

        $periodos = DB::table('sweb_periodos')
            ->select('id', 'periodo', 'label')
            ->orderBy('id')
            ->get();

        $students = DB::table("{$estudiantesTable} as e")
            ->leftJoin('sweb_salones as s', 's.id', '=', 'e.salon_id')
            ->select(
                'e.id',
                'e.contabilidad_id',
                DB::raw("TRIM(CONCAT(COALESCE(e.apellido1,''),' ',COALESCE(e.apellido2,''),' ',COALESCE(e.nombres,''))) as full_name"),
                DB::raw("COALESCE(s.nombre,'') as salon")
            )
            ->where('e.is_active', 1)
            ->when($role === 'docentes' && $sessionUserId > 0, function ($query) use ($annio, $sessionUserId, $reg) {
                $salonIds = $this->getDocenteAllowedSalonIdsForRegistro($annio, $sessionUserId);
                $regSalonId = (int) ($reg->salon_id ?? 0);
                if ($regSalonId > 0 && ! in_array($regSalonId, $salonIds, true)) {
                    $salonIds[] = $regSalonId;
                }
                if (empty($salonIds)) {
                    $query->whereRaw('1 = 0');
                } else {
                    $query->whereIn('e.salon_id', $salonIds);
                }
            })
            ->orderBy('e.apellido1')
            ->orderBy('e.apellido2')
            ->orderBy('e.nombres')
            ->limit(1500)
            ->get();

        return view('admin.observador.registro_academico_edit', [
            'title' => 'Editar Registro Academico',
            'reg' => $reg,
            'periodos' => $periodos,
            'students' => $students,
        ]);
    }

    public function updateRegistroAcademico(Request $request, int $id)
    {
        $reg = DB::table('sweb_estudiantes_reg_desacacom')->where('id', $id)->first();
        if (! $reg) {
            return redirect()->route('admin.observador.registro_academico.index')->with('status', 'No se encontro el registro.');
        }

        $role = Str::lower((string) session('admin_role', ''));
        $sessionUserId = (int) session('admin_user_id', 0);
        abort_unless($this->canAccessRegistroAcademico($role, $sessionUserId), 403);
        $isDocente = $role === 'docentes';
        if ($role === 'docentes' && (int) ($reg->created_by ?? 0) !== $sessionUserId) {
            $isDirectorDelSalon = DB::table('sweb_salones')
                ->where('id', (int) ($reg->salon_id ?? 0))
                ->where('director_id', $sessionUserId)
                ->exists();
            if (! $isDirectorDelSalon) {
                abort(403, 'No tienes permisos para actualizar este registro.');
            }
        }

        $data = $request->validate([
            'annio' => ['required', 'integer', 'min:2000', 'max:2100'],
            'estudiante_id' => ['required', 'integer'],
            'periodo_id' => ['required', 'integer', Rule::exists('sweb_periodos', 'id')],
            'fecha' => ['required', 'date'],
            'fortalezas' => ['nullable', 'string', 'max:1024'],
            'dificultades' => ['nullable', 'string', 'max:1024'],
            'compromisos' => ['nullable', 'string', 'max:1024'],
            'evidencia' => ['nullable', 'file', 'mimes:jpg,jpeg,png,pdf', 'max:5120'],
        ]);

        $fortalezas = trim((string) ($data['fortalezas'] ?? ''));
        $dificultades = trim((string) ($data['dificultades'] ?? ''));
        $compromisos = trim((string) ($data['compromisos'] ?? ''));
        if ($fortalezas === '' && $dificultades === '' && $compromisos === '') {
            return back()->with('status', 'Debes diligenciar fortalezas, dificultades o compromisos.');
        }

        [$estudiantesTable] = $this->resolveStudentTablesByYear((int) $data['annio']);
        $estudiante = DB::table($estudiantesTable)
            ->select('id', 'salon_id')
            ->where('id', (int) $data['estudiante_id'])
            ->first();
        if (! $estudiante) {
            return back()->with('status', 'No se encontro el estudiante seleccionado.');
        }

        $salonId = (int) ($estudiante->salon_id ?? 0);
        $gradoId = (int) (DB::table('sweb_salones')->where('id', $salonId)->value('grado_id') ?? 0);

        if ($isDocente && $sessionUserId > 0) {
            $salonIds = $this->getDocenteAllowedSalonIdsForRegistro((int) $data['annio'], $sessionUserId);
            if (! in_array($salonId, $salonIds, true)) {
                return back()->with('status', 'Solo puedes seleccionar estudiantes de tus cursos asignados o dirigidos.');
            }
        }

        $evidenciaPath = $reg->foto_acudiente ?: '';
        if ($request->hasFile('evidencia')) {
            $evidenciaPath = $this->storeObservacionFile($request->file('evidencia'));
        }

        DB::table('sweb_estudiantes_reg_desacacom')
            ->where('id', $id)
            ->update([
                'estudiante_id' => (int) $data['estudiante_id'],
                'annio' => (int) $data['annio'],
                'periodo_id' => (int) $data['periodo_id'],
                'grado_id' => $gradoId,
                'salon_id' => $salonId,
                'fortalezas' => $fortalezas,
                'dificultades' => $dificultades,
                'compromisos' => $compromisos,
                'fecha' => $data['fecha'],
                'foto_acudiente' => $evidenciaPath,
                'updated_by' => (int) session('admin_user_id', 1),
                'updated_at' => now(),
            ]);

        return redirect()
            ->route('admin.observador.registro_academico.index', ['annio' => (int) $data['annio']])
            ->with('status', 'Guardado correctamente');
    }

    public function deleteRegistroAcademico(int $id)
    {
        $role = Str::lower((string) session('admin_role', ''));
        if (! in_array($role, ['coordinadores', 'admin', 'superadmin'], true)) {
            abort(403, 'No tienes permisos para eliminar registros.');
        }

        $reg = DB::table('sweb_estudiantes_reg_desacacom')
            ->select('id', 'annio')
            ->where('id', $id)
            ->first();

        if (! $reg) {
            return redirect()
                ->route('admin.observador.registro_academico.index')
                ->with('status', 'No se encontro el registro para eliminar.');
        }

        DB::table('sweb_estudiantes_reg_desacacom')->where('id', $id)->delete();

        return redirect()
            ->route('admin.observador.registro_academico.index', ['annio' => (int) ($reg->annio ?? date('Y'))])
            ->with('status', 'Registro eliminado correctamente.');
    }

    private function canAccessRegistroAcademico(string $role, int $userId): bool
    {
        if (in_array($role, ['coordinadores', 'admin', 'superadmin'], true)) {
            return true;
        }

        if ($role === 'docentes' && $userId > 0) {
            return DB::table('sweb_salones')
                ->where('is_active', 1)
                ->where('director_id', $userId)
                ->exists();
        }

        return false;
    }

    private function storeObservacionFile(?UploadedFile $file): string
    {
        if (! $file) {
            return '';
        }

        $folder = public_path('uploads/observador');
        if (! is_dir($folder)) {
            @mkdir($folder, 0775, true);
        }

        $safeName = date('Ymd_His') . '_' . Str::random(10) . '.' . strtolower((string) $file->getClientOriginalExtension());
        $file->move($folder, $safeName);

        return 'uploads/observador/' . $safeName;
    }

    private function getDocenteSalonIdsByAnnio(int $annio, int $docenteId): array
    {
        $assignTable = $this->resolveSalonAsignatProfesorTable($annio);

        return DB::table($assignTable)
            ->where('user_id', $docenteId)
            ->pluck('salon_id')
            ->map(fn ($v) => (int) $v)
            ->filter(fn ($v) => $v > 0)
            ->unique()
            ->values()
            ->all();
    }

    private function getDocenteAllowedSalonIdsForRegistro(int $annio, int $docenteId): array
    {
        $asignados = $this->getDocenteSalonIdsByAnnio($annio, $docenteId);
        $dirigidos = DB::table('sweb_salones')
            ->where('director_id', $docenteId)
            ->where('is_active', 1)
            ->pluck('id')
            ->map(fn ($v) => (int) $v)
            ->filter(fn ($v) => $v > 0)
            ->values()
            ->all();

        return collect(array_merge($asignados, $dirigidos))
            ->map(fn ($v) => (int) $v)
            ->filter(fn ($v) => $v > 0)
            ->unique()
            ->values()
            ->all();
    }

    private function extractPeriodNumber(string $periodo): ?int
    {
        if (preg_match('/(\d+)/', $periodo, $m)) {
            return (int) $m[1];
        }

        return null;
    }

    private function getNextIndicadorCodigo(int $annio, int $periodoId, int $gradoId, int $asignaturaId, string $valorativo): int
    {
        $val = Str::lower(trim($valorativo));

        if (str_contains($val, 'debil')) {
            $min = 200;
            $max = 299;
        } elseif (str_contains($val, 'recom')) {
            $min = 300;
            $max = 399;
        } else {
            $min = 100;
            $max = 199;
        }

        $last = DB::table('sweb_indicadores')
            ->where('annio', $annio)
            ->where('periodo_id', $periodoId)
            ->where('grado_id', $gradoId)
            ->where('asignatura_id', $asignaturaId)
            ->whereRaw('codigo REGEXP "^[0-9]+$"')
            ->whereRaw('CAST(codigo AS UNSIGNED) BETWEEN ? AND ?', [$min, $max])
            ->max(DB::raw('CAST(codigo AS UNSIGNED)'));

        $next = (int) $last + 1;
        if ($next < $min) {
            $next = $min;
        }
        if ($next > $max) {
            $next = $max;
        }

        return $next;
    }

    private function resolveSalonAsignatProfesorTable(int $annio): string
    {
        $yearTable = 'sweb_salon_asignat_profesor_' . $annio;
        $hasYearTable = DB::table('information_schema.tables')
            ->whereRaw('table_schema = DATABASE()')
            ->where('table_name', $yearTable)
            ->exists();

        return $hasYearTable ? $yearTable : 'sweb_salon_asignat_profesor';
    }

    private function resolveStudentTablesByYear(int $annio): array
    {
        $yearStudents = 'sweb_estudiantes_' . $annio;
        $yearDatos = 'sweb_datosestud_' . $annio;

        $hasYearStudents = Cache::remember("has_table_{$yearStudents}", 3600, function () use ($yearStudents) {
            return DB::table('information_schema.tables')
                ->whereRaw('table_schema = DATABASE()')
                ->where('table_name', $yearStudents)
                ->exists();
        });

        $hasYearDatos = Cache::remember("has_table_{$yearDatos}", 3600, function () use ($yearDatos) {
            return DB::table('information_schema.tables')
                ->whereRaw('table_schema = DATABASE()')
                ->where('table_name', $yearDatos)
                ->exists();
        });

        return [
            $hasYearStudents ? $yearStudents : 'sweb_estudiantes',
            $hasYearDatos ? $yearDatos : 'sweb_datosestud',
        ];
    }

}



